<?php
/*
Plugin Name: Elementor Forms Statistics
Requires Plugins: elementor/elementor.php
Plugin URI: https://www.medienproduktion.biz/elementor-forms-extras/
Description: This plugin allows editors to view submissions received through Elementor forms. Additionally, a separate menu provides statistical analyses of the submissions, displayed as charts and tables.
Version: 1.8
Author: Nikolaos Karapanagiotidis
Author URI: https://www.medienproduktion.biz
Text Domain: elementor-forms-statistics

=== Changelog ===

= Version 1.8 – 20. Dezember 2025 =
* Die automatische Bereinigung lässt sich jetzt per Dropdown (Deaktiviert | 1 Stunde | 1 Tag | 1 Monat | 1 Jahr) steuern, die Daten werden weiterhin erst nach Archivierung gelöscht.

= Version 1.7 – 19. Dezember 2025 =
* Neue Bereinigungsoption für Elementor Submissions: Nur wenn die Warnung aktiviert ist, werden alte Einträge nach dem gewählten Stunden-/Tage-/Monate-Intervall gelöscht.

= Version 1.6 – 19. Dezember 2025 =
* Neue Standardfarbpalette, die auf dem aktuellen Design basiert, und das Chart-Footer-Link-Layout passt sich jetzt exakt der Breite von Diagramm/Tabelle an.

= Version 1.0 – 19. Dezember 2025 =
* Neue Option im E-Mail Versand: HTML-Export kann weiterhin gespeichert werden, muss aber nicht mehr zwangsläufig als Anhang versendet werden.

= Version 2.0 - 26. August 2025 =
* Filtern über Checkboxen statt Menü. So kann man einzelne Formulare aus der Statistik ausblenden. 


= Version 1.9 - 28. April 2025 =
* Kurve geht nur bis zum aktuellen Datum


= Version 1.8 - 25. März 2025 =
* Farbschema und Transparenzen im Kurvendiagramm geändert

= Version 1.5 - 1. September 2024 =
* Hinzugefügt: Grafik filter nach Formular
*/

if (!defined('MDP_ARCHIVE_TABLE_SCHEMA_VERSION')) {
    define('MDP_ARCHIVE_TABLE_SCHEMA_VERSION', 1);
}

function mdp_get_archive_table_name() {
    global $wpdb;
    return $wpdb->prefix . 'mdp_form_stats';
}

function mdp_install_archive_table() {
    global $wpdb;
    $table_name = mdp_get_archive_table_name();
    $current_version = (int) get_option('mdp_efs_archive_schema_version', 0);
    if (mdp_archive_table_exists() && $current_version === MDP_ARCHIVE_TABLE_SCHEMA_VERSION) {
        return;
    }
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE {$table_name} (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        form_id varchar(190) NOT NULL,
        form_title text,
        year smallint(4) NOT NULL,
        month tinyint(2) NOT NULL,
        total int(11) NOT NULL DEFAULT 0,
        updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY  (id),
        UNIQUE KEY form_month (form_id(150),year,month)
    ) {$charset_collate};";

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    dbDelta($sql);
}

register_uninstall_hook(__FILE__, 'mdp_uninstall_plugin');

function mdp_activate_plugin() {
    mdp_install_archive_table();
    update_option('mdp_efs_show_archive_notice', 1);
    mdp_schedule_submission_cleanup(true);
    mdp_archive_sync_new_entries(true);
}
register_activation_hook(__FILE__, 'mdp_activate_plugin');
add_action('plugins_loaded', 'mdp_install_archive_table');
add_action('init', 'mdp_archive_maybe_sync');
add_action('admin_post_mdp_run_archive_import', 'mdp_handle_archive_import');

function mdp_archive_table_exists() {
    global $wpdb;
    $table_name = mdp_get_archive_table_name();
    $like = $wpdb->prepare("SHOW TABLES LIKE %s", $table_name);
    return $wpdb->get_var($like) === $table_name;
}

function mdp_archive_has_data() {
    if (!mdp_archive_table_exists()) {
        return false;
    }
    global $wpdb;
    $table = mdp_get_archive_table_name();
    $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} LIMIT 1");
    return $count > 0;
}

function mdp_should_use_archive() {
    if (!mdp_archive_table_exists()) {
        return false;
    }
    if (get_option('mdp_efs_archive_initialized')) {
        return true;
    }
    if (mdp_archive_has_data()) {
        update_option('mdp_efs_archive_initialized', 1);
        return true;
    }
    return false;
}

function mdp_get_archived_form_titles() {
    global $wpdb;
    if (!mdp_archive_table_exists()) {
        return [];
    }
    $table = mdp_get_archive_table_name();
    return $wpdb->get_results("SELECT form_id, MAX(form_title) AS form_title FROM {$table} GROUP BY form_id");
}

function mdp_archive_sync_new_entries($force_full = false) {
    global $wpdb;
    mdp_install_archive_table();
    $archive_table = mdp_get_archive_table_name();
    $source_table = $wpdb->prefix . 'e_submissions';
    $last_id = $force_full ? 0 : (int) get_option('mdp_efs_archive_last_id', 0);

    if ($force_full) {
        $wpdb->query("TRUNCATE TABLE {$archive_table}");
    }

    $where = "s.status NOT LIKE '%trash%'";
    if ($last_id > 0) {
        $where .= $wpdb->prepare(" AND s.ID > %d", $last_id);
    }
    $where .= mdp_get_email_exclusion_clause('s');

    $sql = "
        SELECT s.element_id AS form_id,
               MAX(s.referer_title) AS form_title,
               YEAR(s.created_at_gmt) AS year_value,
               MONTH(s.created_at_gmt) AS month_value,
               COUNT(*) AS total
        FROM {$source_table} s
        WHERE {$where}
        GROUP BY s.element_id, year_value, month_value
    ";
    $rows = $wpdb->get_results($sql);

    if ($rows) {
        foreach ($rows as $row) {
            if (empty($row->form_id)) {
                continue;
            }
            $form_title = $row->form_title ? sanitize_text_field($row->form_title) : '';
            $wpdb->query($wpdb->prepare(
                "INSERT INTO {$archive_table} (form_id, form_title, year, month, total)
                 VALUES (%s, %s, %d, %d, %d)
                 ON DUPLICATE KEY UPDATE total = total + VALUES(total), form_title = CASE WHEN VALUES(form_title) = '' THEN form_title ELSE VALUES(form_title) END",
                $row->form_id,
                $form_title,
                (int) $row->year_value,
                (int) $row->month_value,
                (int) $row->total
            ));
        }
        $max_id_sql = "SELECT MAX(ID) FROM {$source_table} WHERE status NOT LIKE '%trash%'";
        if ($last_id > 0) {
            $max_id_sql .= $wpdb->prepare(" AND ID > %d", $last_id);
        }
        $max_id = (int) $wpdb->get_var($max_id_sql);
        if ($force_full && !$max_id) {
            $max_id = (int) $wpdb->get_var("SELECT MAX(ID) FROM {$source_table}");
        }
        if ($max_id) {
            update_option('mdp_efs_archive_last_id', max($max_id, $last_id));
        }
        update_option('mdp_efs_archive_last_run', current_time('mysql'));
        update_option('mdp_efs_archive_initialized', 1);
    } elseif ($force_full) {
        update_option('mdp_efs_archive_last_run', current_time('mysql'));
        update_option('mdp_efs_archive_initialized', 1);
    }

    update_option('mdp_efs_archive_last_sync', time());
    return count($rows);
}

function mdp_archive_maybe_sync() {
    if (!get_option('mdp_efs_archive_initialized')) {
        return;
    }
    $last_sync = (int) get_option('mdp_efs_archive_last_sync', 0);
    if ($last_sync && (time() - $last_sync) < HOUR_IN_SECONDS) {
        return;
    }
    mdp_archive_sync_new_entries(false);
}

function mdp_handle_archive_import() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('Sie haben keine Berechtigung, diese Seite zu sehen.', 'elementor-forms-statistics'));
    }
    if (!isset($_POST['mdp_archive_nonce']) || !wp_verify_nonce($_POST['mdp_archive_nonce'], 'mdp_archive_import')) {
        wp_die(__('Ungültige Anfrage.', 'elementor-forms-statistics'));
    }
    $initialized = (bool) get_option('mdp_efs_archive_initialized');
    $force_full = !$initialized || !empty($_POST['mdp_archive_reset']);
    mdp_archive_sync_new_entries($force_full);
    $redirect = wp_get_referer();
    if (!$redirect) {
        $redirect = admin_url('admin.php?page=statistiken-archiv');
    }
    $status = $force_full ? 'initialized' : 'synced';
    wp_safe_redirect(add_query_arg('mdp_archive_status', $status, $redirect));
    exit;
}

/* CSS */
function enqueue_custom_plugin_styles() {
    wp_enqueue_style('custom-plugin-styles', plugin_dir_url(__FILE__) . 'assets/css/style.css');
}
add_action('admin_enqueue_scripts', 'enqueue_custom_plugin_styles');


function custom_menu_item() {
    add_menu_page(
        __('Statistik', 'elementor-forms-statistics'), // The title of the menu item
        __('Statistik', 'elementor-forms-statistics'), // The name of the menu item in the navigation bar
        'edit_posts', // Required capability to see the menu item (for editors and admins)
        'statistiken', // A neutral slug for the menu item
        'custom_menu_callback', // The function to call when the menu item is selected
        'dashicons-chart-line', // The icon for the menu item
        2 // The position of the menu item in the navigation bar
    );

    add_submenu_page(
        'statistiken',
        __('Einstellungen', 'elementor-forms-statistics'),
        __('Einstellungen', 'elementor-forms-statistics'),
        'edit_posts',
        'statistiken-einstellungen',
        'mdp_settings_page_callback'
    );

    add_submenu_page(
        'statistiken',
        __('E-Mail Versand', 'elementor-forms-statistics'),
        __('E-Mail Versand', 'elementor-forms-statistics'),
        'edit_posts',
        'statistiken-emailversand',
        'mdp_email_settings_page_callback'
    );

    add_submenu_page(
        'statistiken',
        __('Status', 'elementor-forms-statistics'),
        __('Status', 'elementor-forms-statistics'),
        'edit_posts',
        'statistiken-archiv',
        'mdp_archive_page_callback'
    );
}

function mdp_archive_setup_notice() {
    if (!current_user_can('edit_posts')) {
        return;
    }
    $show = get_option('mdp_efs_show_archive_notice', 0);
    if (get_option('mdp_efs_archive_initialized')) {
        return;
    }
    if (!$show) {
        return;
    }
    $settings_url = admin_url('admin.php?page=statistiken-archiv');
    $message = '<strong>' . __('Elementor Forms Statistics.', 'elementor-forms-statistics') . '</strong> ';
    $message .= __('Bitte Plugin einmalig initialisieren, damit die Statistikdaten gesichert und anschließend automatisch synchronisiert werden.', 'elementor-forms-statistics') . ' ';
    $message .= sprintf(
        __('Öffne die %s, um den Vorgang manuell zu starten und die Sicherung zu prüfen.', 'elementor-forms-statistics'),
        '<a href="' . esc_url($settings_url) . '">' . esc_html__('Einstellungen zur Statistik-Sicherung', 'elementor-forms-statistics') . '</a>'
    );
    $dismiss_url = wp_nonce_url(add_query_arg('mdp_archive_notice_dismiss', '1', admin_url('admin.php')), 'mdp_archive_notice_dismiss');
    echo '<div class="notice notice-info">';
    echo '<p>' . $message . '</p>';
    echo '<p><a href="' . esc_url($dismiss_url) . '">' . esc_html__('Hinweis ausblenden', 'elementor-forms-statistics') . '</a></p>';
    echo '</div>';
}

function mdp_handle_archive_notice_dismiss() {
    if (!current_user_can('edit_posts')) {
        return;
    }
    if (empty($_GET['mdp_archive_notice_dismiss'])) {
        return;
    }
    if (empty($_GET['_wpnonce']) || !wp_verify_nonce(wp_unslash($_GET['_wpnonce']), 'mdp_archive_notice_dismiss')) {
        return;
    }
    update_option('mdp_efs_show_archive_notice', 0);
}
add_action('admin_notices', 'mdp_archive_setup_notice');
add_action('admin_init', 'mdp_handle_archive_notice_dismiss');
add_action('admin_menu', 'custom_menu_item');
add_action('admin_post_mdp_export_stats_html', 'mdp_export_stats_html');
add_action('init', 'mdp_schedule_submission_cleanup');
add_action('admin_post_mdp_send_stats_now', 'mdp_send_stats_now_handler');
add_filter('cron_schedules', 'mdp_add_custom_cron_schedules');
add_action('init', 'mdp_maybe_schedule_stats_email');
add_action('mdp_send_stats_email', 'mdp_send_stats_email_callback');

function mdp_export_stats_html() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('Sie haben keine Berechtigung, diese Seite zu sehen.', 'elementor-forms-statistics'));
    }

    $is_inline_request = isset($_REQUEST['inline']) && $_REQUEST['inline'];
    if ($is_inline_request) {
        $nonce = isset($_REQUEST['mdp_inline_nonce']) ? sanitize_text_field($_REQUEST['mdp_inline_nonce']) : '';
        if (!wp_verify_nonce($nonce, 'mdp_export_html_inline')) {
            wp_die(__('Ungültige Anfrage.', 'elementor-forms-statistics'));
        }
    } else {
        if (!isset($_POST['mdp_export_nonce']) || !wp_verify_nonce($_POST['mdp_export_nonce'], 'mdp_export_html')) {
            wp_die(__('Ungültige Anfrage.', 'elementor-forms-statistics'));
        }
    }

    $html = custom_menu_callback(true, array(
        'export' => true,
        'inline_styles' => true,
        'include_html_document' => true,
        'include_chartjs' => true,
    ));

    $filename = 'anfragen-statistik-' . date('Y-m-d') . '.html';
    nocache_headers();
    $disposition = (isset($_REQUEST['inline']) && $_REQUEST['inline']) ? 'inline' : 'attachment';
    header('Content-Type: text/html; charset=' . get_option('blog_charset'));
    header('Content-Disposition: ' . $disposition . '; filename="' . $filename . '"');
    header('Content-Length: ' . strlen($html));
    echo $html;
    exit;
}

// JS einfügen
function mdp_enqueue_chartjs($hook_suffix) {
    if ($hook_suffix !== 'toplevel_page_statistiken') {
        return;
    }
    wp_enqueue_script(
        'mdp-chartjs',
        plugins_url('/assets/js/chart.js', __FILE__),
        array(),
        '3.0.0',
        false
    );
}
add_action('admin_enqueue_scripts', 'mdp_enqueue_chartjs');

function mdp_get_plugin_version_string() {
    static $cached = null;
    if ($cached !== null) {
        return $cached;
    }
    if (!function_exists('get_plugin_data')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    $plugin_data = get_plugin_data(__FILE__, false, false);
    $cached = isset($plugin_data['Version']) ? $plugin_data['Version'] : '';
    return $cached;
}

function custom_menu_callback($return_output = false, $render_options = array()) {
    $render_defaults = array(
        'export' => false,
        'inline_styles' => false,
        'include_html_document' => false,
        'include_chartjs' => false,
        'show_inline_button' => true,
        'render_chart_as_image' => false,
    );
    $render_options = wp_parse_args($render_options, $render_defaults);
    if (!$return_output && is_admin()) {
        $render_options['show_inline_button'] = false;
    }

    $forms_indexed = mdp_get_forms_indexed();
    $selected_form_ids = mdp_get_selected_form_ids();
    $archive_ready = mdp_should_use_archive();
    $data_source = $archive_ready ? 'archive' : 'live';

    $dataset = mdp_collect_stats_dataset($data_source, array(
        'selected_form_ids' => $selected_form_ids,
        'forms_indexed' => $forms_indexed,
    ));

    if ($archive_ready && empty($dataset['years_desc'])) {
        $archive_ready = false;
        $data_source = 'live';
        $dataset = mdp_collect_stats_dataset($data_source, array(
            'selected_form_ids' => $selected_form_ids,
            'forms_indexed' => $forms_indexed,
        ));
    }

    $forms = $dataset['forms'];
    $form_order = $dataset['form_order'];
    $years_desc = $dataset['years_desc'];
    $years_asc = $dataset['years_asc'];
    $year_totals = $dataset['year_totals'];
    $current_year = $dataset['current_year'];
    $current_month = $dataset['current_month'];

    $month_template = mdp_get_empty_month_template();
    $month_labels = array(
        1 => __('Januar', 'elementor-forms-statistics'),
        2 => __('Februar', 'elementor-forms-statistics'),
        3 => __('März', 'elementor-forms-statistics'),
        4 => __('April', 'elementor-forms-statistics'),
        5 => __('Mai', 'elementor-forms-statistics'),
        6 => __('Juni', 'elementor-forms-statistics'),
        7 => __('Juli', 'elementor-forms-statistics'),
        8 => __('August', 'elementor-forms-statistics'),
        9 => __('September', 'elementor-forms-statistics'),
        10 => __('Oktober', 'elementor-forms-statistics'),
        11 => __('November', 'elementor-forms-statistics'),
        12 => __('Dezember', 'elementor-forms-statistics'),
    );

    $color_slots = mdp_get_curve_color_slots();
    $slot_count = count($color_slots);

    $reference_year = !empty($years_asc) ? (int) max($years_asc) : (int) date('Y');
    $chart_datasets = array();
    foreach ($years_asc as $index => $year) {
        $year_diff = max(0, $reference_year - (int) $year);
        $slot_index = min($year_diff, $slot_count - 1);
        $slot = $color_slots[$slot_index];
        $border_color = mdp_hex_to_rgba($slot['color'], $slot['alpha']);
        $fill_color = mdp_hex_to_rgba($slot['color'], max(0.05, $slot['alpha'] * 0.45));
        $data_points = array();
        foreach ($month_labels as $month_index => $label) {
            $value = isset($year_totals[$year][$month_index]) ? (int) $year_totals[$year][$month_index] : 0;
            $data_points[] = $value;
        }
        $chart_datasets[] = array(
            'label' => (string) $year,
            'data' => $data_points,
            'borderColor' => $border_color,
            'pointBackgroundColor' => $border_color,
            'pointBorderColor' => $border_color,
            'borderWidth' => 3,
            'backgroundColor' => $fill_color,
            'fill' => true,
        );
    }
    $datasets_json = wp_json_encode($chart_datasets);
    $has_chart = !empty($chart_datasets);

    $has_data = !empty($form_order) && !empty($years_desc);

    ob_start();

    if ($render_options['include_html_document']) {
        echo '<!DOCTYPE html><html><head>';
        echo '<meta charset="' . esc_attr(get_bloginfo('charset')) . '">';
        echo '<title>' . esc_html__('Anfragen Statistik', 'elementor-forms-statistics') . '</title>';
        echo '<meta name="robots" content="noindex,nofollow,noarchive">';
        if ($render_options['inline_styles']) {
            echo '<style>' . mdp_get_inline_export_styles() . '</style>';
        }
        echo '</head><body class="mdp-export-body">';
    } else {
        if ($render_options['inline_styles']) {
            echo '<style>' . mdp_get_inline_export_styles() . '</style>';
        }
    }

    $site_url_display = esc_html(preg_replace('#^https?://#', '', home_url()));
    echo '<div class="wrap mdp-stats-root">';
    echo '<h1>' . sprintf(esc_html__('Statistik %s', 'elementor-forms-statistics'), $site_url_display) . '</h1>';
    if (!$render_options['export'] && $render_options['show_inline_button']) {
        $inline_url = wp_nonce_url(
            add_query_arg(
                array(
                    'action' => 'mdp_export_stats_html',
                    'inline' => 1,
                ),
                admin_url('admin-post.php')
            ),
            'mdp_export_html_inline',
            'mdp_inline_nonce'
        );
        echo '<div class="mdp-inline-view">';
        echo '<span class="mdp-inline-text">' . esc_html__('Für Chartansicht im Browser ansehen', 'elementor-forms-statistics') . '</span>';
        echo '<a class="button button-primary mdp-inline-button" href="' . esc_url($inline_url) . '" target="_blank" rel="noopener noreferrer">';
        echo esc_html__('Im Browser anzeigen', 'elementor-forms-statistics');
        echo '</a></div>';
    }

    if ($has_chart) {
        if ($render_options['render_chart_as_image']) {
            $chart_image_data = mdp_generate_chart_image_base64($chart_datasets, $current_year, $current_month);
            if ($chart_image_data) {
                echo '<div class="mdp-chart mdp-chart-image"><img src="' . esc_attr($chart_image_data) . '" alt="' . esc_attr__('Diagramm der Formularanfragen', 'elementor-forms-statistics') . '"></div>';
            } else {
                echo '<p>' . esc_html__('Das Diagramm konnte nicht als Bild gerendert werden.', 'elementor-forms-statistics') . '</p>';
            }
        } else {
            echo '<div class="mdp-chart"><canvas id="myChart"></canvas></div>';
        }
    } else {
        echo '<p>' . esc_html__('Keine Daten vorhanden.', 'elementor-forms-statistics') . '</p>';
    }

    $plugin_version = mdp_get_plugin_version_string();
    $footer_text = sprintf(__('%s · 2025 · v%s', 'elementor-forms-statistics'), __('Elementor Forms Statistics', 'elementor-forms-statistics'), $plugin_version);
    $footer_link = '<a href="https://www.medienproduktion.biz/elementor-forms-statistics/" target="_blank" rel="noopener noreferrer">' . esc_html($footer_text) . '</a>';
    echo '<div class="mdp-chart-footer">' . $footer_link . '</div>';

    if ($has_data) {
        foreach ($years_desc as $year) {
            echo '<h2 class="mdp-year-heading">' . esc_html($year) . '</h2>';
            echo '<div class="mdp-table-wrapper">';
            echo '<table class="mdp-table"><thead><tr>';
            echo '<th>' . esc_html__('Seite', 'elementor-forms-statistics') . '</th>';
            foreach ($month_labels as $month_label) {
                echo '<th>' . esc_html($month_label) . '</th>';
            }
            echo '<th>' . esc_html__('Summe', 'elementor-forms-statistics') . '</th>';
            echo '</tr></thead><tbody>';

            foreach ($form_order as $form_id) {
                if (!isset($forms[$form_id])) {
                    continue;
                }
                $form_data = $forms[$form_id];
                $current_months = isset($form_data['yearly'][$year]) ? $form_data['yearly'][$year] : $month_template;
                $previous_year_months = isset($form_data['yearly'][$year - 1]) ? $form_data['yearly'][$year - 1] : $month_template;
                $has_previous_year_data = isset($form_data['yearly'][$year - 1]);
                $annual_total = 0;
                $previous_annual_total = 0;
                foreach ($current_months as $month_value) {
                    $annual_total += (int) $month_value;
                }
                foreach ($previous_year_months as $prev_value) {
                    $previous_annual_total += (int) $prev_value;
                }

                echo '<tr>';
                echo '<td>' . esc_html($form_data['title']) . '</td>';

                foreach ($month_labels as $month_index => $month_label) {
                    $value = isset($current_months[$month_index]) ? (int) $current_months[$month_index] : 0;
                    $previous_value = isset($previous_year_months[$month_index]) ? (int) $previous_year_months[$month_index] : 0;
                    $is_past_period = ($year < $current_year) || ($year === $current_year && $month_index < $current_month);
                    $cell_style = $is_past_period ? mdp_get_comparison_background($value, $previous_value, $has_previous_year_data) : '';
                    $emphasize_change = $is_past_period ? mdp_should_emphasize_change($value, $previous_value, $has_previous_year_data) : false;

                    if ($emphasize_change) {
                        $content = '<strong>' . ($value === 0 ? '0' : esc_html($value)) . '</strong>';
                    } elseif ($value === 0) {
                        $content = '&nbsp;';
                    } else {
                        $content = esc_html($value);
                    }
                    echo '<td' . $cell_style . '>' . $content . '</td>';
                }

                $annual_is_past = ($year < $current_year);
                $annual_cell_style = $annual_is_past ? mdp_get_comparison_background($annual_total, $previous_annual_total, $has_previous_year_data) : '';
                $annual_emphasis = $annual_is_past ? mdp_should_emphasize_change($annual_total, $previous_annual_total, $has_previous_year_data) : false;
                $annual_content = $annual_emphasis ? '<strong>' . esc_html($annual_total) . '</strong>' : esc_html($annual_total);
                echo '<td' . $annual_cell_style . '>' . $annual_content . '</td>';
                echo '</tr>';
            }

            $current_year_totals = isset($year_totals[$year]) ? $year_totals[$year] : $month_template;
            $previous_year_totals = isset($year_totals[$year - 1]) ? $year_totals[$year - 1] : $month_template;
            $has_previous_year_totals = isset($year_totals[$year - 1]);
            $grand_total = array_sum($current_year_totals);
            $previous_grand_total = array_sum($previous_year_totals);

            echo '<tr>';
            echo '<td><strong>' . esc_html__('Gesamt', 'elementor-forms-statistics') . '</strong></td>';
            foreach ($month_labels as $month_index => $month_label) {
                $value = isset($current_year_totals[$month_index]) ? (int) $current_year_totals[$month_index] : 0;
                $previous_value = isset($previous_year_totals[$month_index]) ? (int) $previous_year_totals[$month_index] : 0;
                $is_past_period = ($year < $current_year) || ($year === $current_year && $month_index < $current_month);
                $cell_style = $is_past_period ? mdp_get_comparison_background($value, $previous_value, $has_previous_year_totals) : '';
                $emphasize_change = $is_past_period ? mdp_should_emphasize_change($value, $previous_value, $has_previous_year_totals) : false;
                $content_value = $emphasize_change ? '<strong>' . esc_html($value) . '</strong>' : '<strong>' . esc_html($value) . '</strong>';
                echo '<td' . $cell_style . '>' . $content_value . '</td>';
            }
            $grand_is_past = ($year < $current_year);
            $grand_cell_style = $grand_is_past ? mdp_get_comparison_background($grand_total, $previous_grand_total, $has_previous_year_totals) : '';
            $grand_emphasis = $grand_is_past ? mdp_should_emphasize_change($grand_total, $previous_grand_total, $has_previous_year_totals) : false;
            $grand_content = $grand_emphasis ? '<strong>' . esc_html($grand_total) . '</strong>' : '<strong>' . esc_html($grand_total) . '</strong>';
            echo '<td' . $grand_cell_style . '>' . $grand_content . '</td>';
            echo '</tr>';

            echo '</tbody></table>';
            echo '</div>';
        }
    }

    echo '</div>';

    if ($has_chart && $render_options['include_chartjs'] && !$render_options['render_chart_as_image']) {
        $chart_js_path = plugin_dir_path(__FILE__) . 'assets/js/chart.js';
        if (file_exists($chart_js_path)) {
            $chart_js_content = file_get_contents($chart_js_path);
            if ($chart_js_content !== false) {
                echo '<script>' . $chart_js_content . '</script>';
            }
        }
    }

    if ($has_chart && !$render_options['render_chart_as_image']) {
        ?>
        <script>
        (function() {
            var canvas = document.getElementById('myChart');
            if (!canvas) {
                return;
            }
            var ctx = canvas.getContext('2d');
            var chartData = <?php echo $datasets_json; ?>;
            var currentYear = <?php echo (int) $current_year; ?>;
            var currentMonthIndex = <?php echo (int) $current_month - 1; ?>;

            chartData.forEach(function(dataset) {
                if (parseInt(dataset.label, 10) === currentYear) {
                    dataset.segment = dataset.segment || {};
                    dataset.segment.borderDash = function(ctx) {
                        return ctx.p1DataIndex === currentMonthIndex ? [6, 4] : undefined;
                    };
                }
            });

            var labels = [];
            for (var month = 1; month <= 12; month++) {
                labels.push(month);
            }

            var data = {
                labels: labels,
                datasets: chartData
            };

            var options = {
                responsive: true,
                maintainAspectRatio: false,
                tooltips: {
                    callbacks: {
                        title: function() {
                            return '<?php echo esc_js(__('Anzahl Anfragen', 'elementor-forms-statistics')); ?>';
                        },
                        label: function(tooltipItem) {
                            return tooltipItem.yLabel + ' ' + '<?php echo esc_js(__('Anfragen', 'elementor-forms-statistics')); ?>';
                        }
                    }
                },
                scales: {
                    y: {
                        ticks: {
                            stepSize: 1
                        }
                    }
                }
            };
            options.onHover = function(event, elements) {
                var canvas = this && this.canvas ? this.canvas : (event && event.target ? event.target : null);
                if (canvas) {
                    canvas.style.cursor = elements && elements.length ? 'pointer' : 'default';
                }
            };

            function hexToRgba(hex, alpha) {
                var r = parseInt(hex.substr(1, 2), 16);
                var g = parseInt(hex.substr(3, 2), 16);
                var b = parseInt(hex.substr(5, 2), 16);
                return 'rgba(' + r + ',' + g + ',' + b + ',' + alpha + ')';
            }

            for (var i = 0; i < chartData.length; i++) {
                chartData[i].fill = true;
                chartData[i].tension = 0.35;
                var lineColor = chartData[i].borderColor || '#4b5563';
                if (lineColor.indexOf('rgba') === 0) {
                    var rgbaParts = lineColor.match(/^rgba\((\d+),\s*(\d+),\s*(\d+),\s*(\d+\.?\d*)\)$/);
                    if (rgbaParts) {
                        chartData[i].backgroundColor = 'rgba(' + rgbaParts[1] + ',' + rgbaParts[2] + ',' + rgbaParts[3] + ',0.1)';
                    }
                } else if (lineColor.indexOf('#') === 0) {
                    chartData[i].backgroundColor = hexToRgba(lineColor, 0.1);
                }
            }

            var chartInstance = new Chart(ctx, {
                type: 'line',
                data: data,
                options: options
            });
        })();
        </script>
        <?php
    }

    if ($render_options['include_html_document']) {
        echo '</body></html>';
    }

    $output = ob_get_clean();
    if ($return_output) {
        return $output;
    }
    echo $output;
}

function mdp_hex_to_rgb_array($hex_color) {
    $hex_color = trim($hex_color);
    if ($hex_color === '') {
        return array(0, 0, 0);
    }
    if ($hex_color[0] === '#') {
        $hex_color = substr($hex_color, 1);
    }
    if (strlen($hex_color) === 3) {
        $hex_color = $hex_color[0] . $hex_color[0] . $hex_color[1] . $hex_color[1] . $hex_color[2] . $hex_color[2];
    }
    $int = hexdec($hex_color);
    return array(
        ($int >> 16) & 255,
        ($int >> 8) & 255,
        $int & 255,
    );
}

function mdp_catmull_rom_point($p0, $p1, $p2, $p3, $t, $tension) {
    $t2 = $t * $t;
    $t3 = $t2 * $t;
    $v0x = ($p2['x'] - $p0['x']) * $tension;
    $v0y = ($p2['y'] - $p0['y']) * $tension;
    $v1x = ($p3['x'] - $p1['x']) * $tension;
    $v1y = ($p3['y'] - $p1['y']) * $tension;

    $x = (2 * $p1['x'] - 2 * $p2['x'] + $v0x + $v1x) * $t3
        + (-3 * $p1['x'] + 3 * $p2['x'] - 2 * $v0x - $v1x) * $t2
        + $v0x * $t + $p1['x'];

    $y = (2 * $p1['y'] - 2 * $p2['y'] + $v0y + $v1y) * $t3
        + (-3 * $p1['y'] + 3 * $p2['y'] - 2 * $v0y - $v1y) * $t2
        + $v0y * $t + $p1['y'];

    return array('x' => $x, 'y' => $y);
}

function mdp_generate_spline_points($points, $segments = 12, $tension = 0.35) {
    $count = count($points);
    if ($count < 3) {
        return $points;
    }
    $spline = array();
    for ($i = 0; $i < $count - 1; $i++) {
        $p0 = $points[$i === 0 ? $i : $i - 1];
        $p1 = $points[$i];
        $p2 = $points[$i + 1];
        $p3 = $points[$i + 1 >= $count - 1 ? $count - 1 : $i + 2];

        for ($j = 0; $j < $segments; $j++) {
            $t = $j / $segments;
            $spline[] = mdp_catmull_rom_point($p0, $p1, $p2, $p3, $t, $tension);
        }
    }
    $spline[] = end($points);
    return $spline;
}

function mdp_generate_chart_image_base64($chart_datasets, $current_year, $current_month) {
    if (empty($chart_datasets) || !function_exists('imagecreatetruecolor') || !function_exists('imagepng')) {
        return '';
    }

    $width = 1090;
    $height = 420;
    $padding = 18;
    $chart_left = 70;
    $chart_top = 40;
    $chart_right = $width - 30;
    $chart_bottom = $height - 60;
    $chart_width = $chart_right - $chart_left;
    $chart_height = $chart_bottom - $chart_top;
    $months = 12;
    $month_spacing = $chart_width / ($months - 1);

    $max_value = 0;
    foreach ($chart_datasets as $dataset) {
        if (empty($dataset['data']) || !is_array($dataset['data'])) {
            continue;
        }
        foreach ($dataset['data'] as $value) {
            if ($value > $max_value) {
                $max_value = $value;
            }
        }
    }
    if ($max_value < 1) {
        $max_value = 1;
    }

    $image = imagecreatetruecolor($width, $height);
    imagealphablending($image, true);
    imagesavealpha($image, true);

    $background = imagecolorallocate($image, 231, 234, 239);
    $card_fill = imagecolorallocate($image, 248, 249, 252);
    $card_border = imagecolorallocate($image, 220, 224, 232);
    $grid_color = imagecolorallocate($image, 214, 218, 226);
    $axis_color = imagecolorallocate($image, 170, 178, 192);
    $text_color = imagecolorallocate($image, 29, 35, 39);
    $legend_text_color = imagecolorallocate($image, 50, 57, 65);
    imagefilledrectangle($image, 0, 0, $width - 1, $height - 1, $background);
    imagefilledrectangle($image, $padding, $padding, $width - $padding, $height - $padding, $card_fill);
    imagerectangle($image, $padding, $padding, $width - $padding, $height - $padding, $card_border);
    if (function_exists('imageantialias')) {
        imageantialias($image, true);
    }

    $title = __('Anfragen pro Monat und Jahr', 'elementor-forms-statistics');
    imagestring($image, 5, $chart_left, 12, $title, $text_color);

    $grid_steps = 6;
    $step_value = $max_value / $grid_steps;
    for ($i = 0; $i <= $grid_steps; $i++) {
        $y = $chart_bottom - ($chart_height / $grid_steps) * $i;
        imageline($image, $chart_left, (int) $y, $chart_right, (int) $y, $grid_color);
        $label_value = round($step_value * $i);
        imagestring($image, 2, 20, (int) $y - 6, (string) $label_value, $text_color);
    }

    for ($i = 0; $i < $months; $i++) {
        $x = $chart_left + $month_spacing * $i;
        imageline($image, (int) $x, $chart_top, (int) $x, $chart_bottom, $grid_color);
        imagestring($image, 2, (int) ($x - 5), $chart_bottom + 10, (string) ($i + 1), $text_color);
    }

    imageline($image, $chart_left, $chart_top, $chart_left, $chart_bottom, $axis_color);
    imageline($image, $chart_left, $chart_bottom, $chart_right, $chart_bottom, $axis_color);

    $legend_x = $chart_left;
    $legend_y = 30;
    $legend_gap = 120;

    foreach ($chart_datasets as $dataset) {
        if (empty($dataset['data']) || !is_array($dataset['data'])) {
            continue;
        }
        $rgb = mdp_hex_to_rgb_array(isset($dataset['borderColor']) ? $dataset['borderColor'] : '#2d3748');
        $line_color = imagecolorallocate($image, $rgb[0], $rgb[1], $rgb[2]);
        $fill_color = imagecolorallocatealpha($image, $rgb[0], $rgb[1], $rgb[2], 110);
        imagesetthickness($image, 3);

        imagefilledrectangle($image, $legend_x, $legend_y - 12, $legend_x + 14, $legend_y + 2, $line_color);
        $label = isset($dataset['label']) ? $dataset['label'] : '';
        imagestring($image, 4, $legend_x + 20, $legend_y - 12, (string) $label, $legend_text_color);
        $legend_x += $legend_gap;
        if ($legend_x + $legend_gap > $chart_right) {
            $legend_x = $chart_left;
            $legend_y += 20;
        }

        $points = array();
        foreach ($dataset['data'] as $index => $value) {
            $x = $chart_left + $month_spacing * $index;
            $value = max(0, (float) $value);
            $ratio = $value / $max_value;
            $y = $chart_bottom - ($ratio * $chart_height);
            $points[] = array(
                'x' => $x,
                'y' => $y,
                'month_index' => $index,
            );
        }

        if (count($points) < 2) {
            continue;
        }

        $spline_points = mdp_generate_spline_points($points, 10, 0.35);
        $polygon = array();
        foreach ($spline_points as $pt) {
            $polygon[] = (int) round($pt['x']);
            $polygon[] = (int) round($pt['y']);
        }
        $last_point = end($points);
        $first_point = reset($points);
        $polygon[] = (int) round($last_point['x']);
        $polygon[] = (int) round($chart_bottom);
        $polygon[] = (int) round($first_point['x']);
        $polygon[] = (int) round($chart_bottom);
        imagefilledpolygon($image, $polygon, count($polygon) / 2, $fill_color);

        $prev = null;
        $dash_threshold = $chart_left + $month_spacing * $current_month;
        foreach ($spline_points as $pt) {
            if ($prev) {
                $x1 = (int) round($prev['x']);
                $y1 = (int) round($prev['y']);
                $x2 = (int) round($pt['x']);
                $y2 = (int) round($pt['y']);
                $use_dash = isset($dataset['label']) && (string) $dataset['label'] === (string) $current_year && $prev['x'] >= $dash_threshold;
                if ($use_dash) {
                    imagedashedline($image, $x1, $y1, $x2, $y2, $line_color);
                } else {
                    imageline($image, $x1, $y1, $x2, $y2, $line_color);
                }
            }
            $prev = $pt;
        }
    }

    ob_start();
    imagepng($image);
    $png_data = ob_get_clean();
    imagedestroy($image);
    if ($png_data === false || $png_data === '') {
        return '';
    }
    return 'data:image/png;base64,' . base64_encode($png_data);
}

// Fetch all unique forms using the stable element_id as identifier
function custom_get_all_forms() {
    global $wpdb;
    return $wpdb->get_results(
        "SELECT element_id, MAX(referer_title) AS referer_title 
        FROM " . $wpdb->prefix . "e_submissions 
        WHERE element_id IS NOT NULL AND element_id != '' 
        GROUP BY element_id 
        ORDER BY referer_title"
    );
}

function mdp_get_selected_form_ids() {
    $stored = get_option('mdp_efs_selected_form_ids', []);
    if (is_array($stored)) {
        return array_map('sanitize_text_field', $stored);
    }
    return [];
}

function mdp_get_forms_indexed() {
    $forms = custom_get_all_forms();
    $indexed = [];
    foreach ($forms as $form_entry) {
        if (!empty($form_entry->element_id)) {
            $indexed[$form_entry->element_id] = $form_entry;
        }
    }

    $archived_forms = mdp_get_archived_form_titles();
    if (!empty($archived_forms)) {
        foreach ($archived_forms as $archived) {
            if (empty($archived->form_id)) {
                continue;
            }
            if (!isset($indexed[$archived->form_id]) || empty($indexed[$archived->form_id]->referer_title)) {
                $indexed[$archived->form_id] = (object) array(
                    'element_id' => $archived->form_id,
                    'referer_title' => $archived->form_title,
                );
            }
        }
    }

    return $indexed;
}

function mdp_resolve_form_title($form_id, $form_entry = null) {
    if ($form_entry && !empty($form_entry->referer_title)) {
        return $form_entry->referer_title;
    }

    static $resolved_titles = [];
    if (isset($resolved_titles[$form_id])) {
        return $resolved_titles[$form_id];
    }

    global $wpdb;
    $title = $wpdb->get_var($wpdb->prepare(
        "SELECT referer_title FROM " . $wpdb->prefix . "e_submissions 
         WHERE element_id = %s AND referer_title IS NOT NULL AND referer_title != '' 
         ORDER BY created_at_gmt DESC LIMIT 1",
        $form_id
    ));

    $resolved_titles[$form_id] = $title ? $title : '';
    return $resolved_titles[$form_id];
}

function mdp_get_empty_month_template() {
    return array(
        1 => 0,
        2 => 0,
        3 => 0,
        4 => 0,
        5 => 0,
        6 => 0,
        7 => 0,
        8 => 0,
        9 => 0,
        10 => 0,
        11 => 0,
        12 => 0,
    );
}

function mdp_format_form_title($form_id, $forms_indexed, $fallback = '') {
    if (isset($forms_indexed[$form_id]) && !empty($forms_indexed[$form_id]->referer_title)) {
        return $forms_indexed[$form_id]->referer_title;
    }
    if ($fallback !== '') {
        return $fallback;
    }
    $resolved = mdp_resolve_form_title($form_id);
    if ($resolved !== '') {
        return $resolved;
    }
    return __('Unbenannte Seite', 'elementor-forms-statistics');
}

function mdp_collect_stats_dataset($data_source = 'live', $args = array()) {
    global $wpdb;

    $defaults = array(
        'selected_form_ids' => array(),
        'forms_indexed' => array(),
    );
    $args = wp_parse_args($args, $defaults);
    $selected_ids = array();
    foreach ($args['selected_form_ids'] as $form_id) {
        $form_id = sanitize_text_field($form_id);
        if ($form_id !== '') {
            $selected_ids[] = $form_id;
        }
    }
    $forms_indexed = $args['forms_indexed'];

    $rows = array();
    if ($data_source === 'archive') {
        if (!mdp_archive_table_exists()) {
            $rows = array();
        } else {
            $table = mdp_get_archive_table_name();
            $where = "WHERE 1=1";
            if (!empty($selected_ids)) {
                $placeholders = implode(', ', array_fill(0, count($selected_ids), '%s'));
                $where .= $wpdb->prepare(" AND form_id IN ($placeholders)", $selected_ids);
            }
            $rows = $wpdb->get_results("
                SELECT form_id, MAX(form_title) AS form_title, year AS year_value, month AS month_value, SUM(total) AS total
                FROM {$table}
                {$where}
                GROUP BY form_id, year_value, month_value
                ORDER BY year_value, month_value
            ");
        }
    } else {
        $table = $wpdb->prefix . 'e_submissions';
        $where = "WHERE s.status NOT LIKE '%trash%'";
        if (!empty($selected_ids)) {
            $placeholders = implode(', ', array_fill(0, count($selected_ids), '%s'));
            $where .= $wpdb->prepare(" AND s.element_id IN ($placeholders)", $selected_ids);
        }
        $where .= mdp_get_email_exclusion_clause('s');
        $rows = $wpdb->get_results("
            SELECT s.element_id AS form_id,
                   MAX(s.referer_title) AS form_title,
                   YEAR(s.created_at_gmt) AS year_value,
                   MONTH(s.created_at_gmt) AS month_value,
                   COUNT(*) AS total
            FROM {$table} s
            {$where}
            GROUP BY s.element_id, year_value, month_value
            ORDER BY year_value, month_value
        ");
    }

    $forms = array();
    $form_titles = array();
    $year_totals = array();
    $years_map = array();
    $month_template = mdp_get_empty_month_template();

    foreach ($rows as $row) {
        if (empty($row->form_id)) {
            continue;
        }
        $form_id = sanitize_text_field($row->form_id);
        $year = (int) $row->year_value;
        $month = (int) $row->month_value;
        $total = (int) $row->total;
        if ($year <= 0 || $month <= 0) {
            continue;
        }

        $years_map[$year] = true;
        if (!isset($year_totals[$year])) {
            $year_totals[$year] = $month_template;
        }
        if (!isset($year_totals[$year][$month])) {
            $year_totals[$year][$month] = 0;
        }
        $year_totals[$year][$month] += $total;

        if (!isset($forms[$form_id])) {
            $forms[$form_id] = array(
                'title' => '',
                'yearly' => array(),
            );
        }
        if (!isset($forms[$form_id]['yearly'][$year])) {
            $forms[$form_id]['yearly'][$year] = $month_template;
        }
        $forms[$form_id]['yearly'][$year][$month] += $total;

        if (!isset($form_titles[$form_id]) || $form_titles[$form_id] === '') {
            $form_titles[$form_id] = $row->form_title ? sanitize_text_field($row->form_title) : '';
        }
    }

    $selected_ids = array_unique($selected_ids);
    foreach ($selected_ids as $form_id) {
        if (!isset($forms[$form_id])) {
            $forms[$form_id] = array(
                'title' => '',
                'yearly' => array(),
            );
        }
    }

    foreach ($forms as $form_id => &$form_data) {
        foreach ($form_data['yearly'] as $year => &$months) {
            $months = array_replace($month_template, $months);
        }
        unset($months);
        $fallback = isset($form_titles[$form_id]) ? $form_titles[$form_id] : '';
        $form_data['title'] = mdp_format_form_title($form_id, $forms_indexed, $fallback);
    }
    unset($form_data);

    $form_order = array_keys($forms);
    if (!empty($selected_ids)) {
        $selected_lookup = array_flip($selected_ids);
        $form_order = array_values(array_filter($form_order, function($form_id) use ($selected_lookup) {
            return isset($selected_lookup[$form_id]);
        }));
    }
    usort($form_order, function($a, $b) use ($forms) {
        return strcasecmp($forms[$a]['title'], $forms[$b]['title']);
    });

    $years_asc = array_keys($years_map);
    sort($years_asc);
    $years_desc = array_reverse($years_asc);

    foreach ($year_totals as $year => &$months) {
        $months = array_replace($month_template, $months);
    }
    unset($months);

    return array(
        'forms' => $forms,
        'form_order' => $form_order,
        'years_asc' => $years_asc,
        'years_desc' => $years_desc,
        'year_totals' => $year_totals,
        'current_year' => (int) date('Y'),
        'current_month' => (int) date('n'),
    );
}

function mdp_get_schedule_interval() {
    $value = get_option('mdp_efs_email_interval', 'monthly');
    $valid = array('disabled', 'daily', 'weekly', 'monthly');
    if (!in_array($value, $valid, true)) {
        $value = 'monthly';
    }
    return $value;
}

function mdp_get_email_recipients() {
    $raw = get_option('mdp_efs_email_recipients', '');
    if (!is_string($raw) || $raw === '') {
        return [];
    }
    $parts = preg_split('/[\r\n,]+/', $raw);
    $emails = [];
    foreach ($parts as $part) {
        $trimmed = trim($part);
        if ($trimmed === '') {
            continue;
        }
        $email = sanitize_email($trimmed);
        if ($email) {
            $emails[] = strtolower($email);
        }
    }
    return array_values(array_unique($emails));
}

function mdp_get_email_recipients_text() {
    $emails = mdp_get_email_recipients();
    return implode("\n", $emails);
}

function mdp_get_default_email_message() {
    return __("Hallo,

anbei die aktuelle Übersicht der Formular-Anfragen.
%stats_link%

Deine freundliche Website
%s

", 'elementor-forms-statistics');
}

function mdp_get_email_message() {
    $default = mdp_get_default_email_message();
    $message = get_option('mdp_efs_email_message', $default);
    if (!is_string($message) || $message === '') {
        $message = $default;
    }
    return $message;
}

function mdp_get_email_subject_template() {
    $template = get_option('mdp_efs_email_subject', '📈 ' . __('Anfragen Statistik – %s', 'elementor-forms-statistics'));
    if (!is_string($template) || $template === '') {
        $template = '📈 ' . __('Anfragen Statistik – %s', 'elementor-forms-statistics');
    }
    return $template;
}

function mdp_should_include_stats_attachment() {
    $value = get_option('mdp_efs_include_attachment', '1');
    return $value !== '0';
}

function mdp_build_email_subject($date_string = '') {
    $template = mdp_get_email_subject_template();
    $formatted_date = $date_string !== '' ? $date_string : date_i18n('F Y');
    if (strpos($template, '%s') !== false) {
        return sprintf($template, $formatted_date);
    }
    return trim($template . ' ' . $formatted_date);
}

function mdp_build_styled_email_body($message_with_link, $link_note, $export_url, $link_label) {
    $message_html = wpautop($message_with_link);
    $badge = '';

    $note_block = $link_note ? '<div style="margin:24px 0 0;padding:0;">' . $link_note . '</div>' : '';
    $branding_url = 'https://www.medienproduktion.biz/elementor-forms-statistics/';
    $plugin_version = mdp_get_plugin_version_string();
    $brand_text = sprintf(__('%s · 2025 · v%s', 'elementor-forms-statistics'), __('Elementor Forms Statistics', 'elementor-forms-statistics'), $plugin_version);
    $branding_block = '<p style="margin:20px 0 0;font-size:11px;color:#9ca3b4;text-align:center;">'
        . '<a href="' . esc_url($branding_url) . '" target="_blank" rel="noopener noreferrer" style="color:#9ca3b4;text-decoration:none;">'
        . esc_html($brand_text)
        . '</a></p>';
    return '
    <div style="background:#f5f6fb;padding:48px 16px;font-family:-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,Helvetica Neue,Arial,sans-serif;color:#1f2a44;">
        <div style="max-width:720px;margin:0 auto;background:#ffffff;border-radius:20px;box-shadow:0 25px 60px rgba(31,42,68,0.12);padding:40px 48px;">
            ' . ($badge ? '<p style="margin:0 0 6px;font-size:12px;font-weight:600;letter-spacing:0.2em;text-transform:uppercase;color:#8c96b0;">' . esc_html($badge) . '</p>' : '') . '
            <div style="font-size:16px;line-height:1.7;color:#2d3756;">' . $message_html . '</div>
            ' . $note_block . '
            ' . $branding_block . '
        </div>
    </div>';
}

function mdp_get_default_curve_color_slots() {
    return array(
        array('color' => '#a4bbb0', 'alpha' => 0.95),
        array('color' => '#c9b1a5', 'alpha' => 0.9),
        array('color' => '#d49484', 'alpha' => 0.88),
        array('color' => '#edc1a6', 'alpha' => 0.85),
        array('color' => '#bec0b2', 'alpha' => 0.82),
        array('color' => '#d7b69f', 'alpha' => 0.8),
        array('color' => '#8fa6bf', 'alpha' => 0.77),
        array('color' => '#9dc1c8', 'alpha' => 0.74),
        array('color' => '#c6d5dd', 'alpha' => 0.7),
        array('color' => '#d7d6d1', 'alpha' => 0.66),
    );
}

function mdp_get_curve_color_slots() {
    $stored = get_option('mdp_efs_curve_color_slots');
    $defaults = mdp_get_default_curve_color_slots();
    $slots = array();
    for ($i = 0; $i < 10; $i++) {
        $slot = isset($stored[$i]) && is_array($stored[$i]) ? $stored[$i] : array();
        $color = isset($slot['color']) ? sanitize_hex_color($slot['color']) : '';
        $alpha = isset($slot['alpha']) ? (float) $slot['alpha'] : $defaults[$i]['alpha'];
        if ($color === '') {
            $color = $defaults[$i]['color'];
        }
        if ($alpha < 0) {
            $alpha = 0;
        } elseif ($alpha > 1) {
            $alpha = 1;
        }
        $slots[] = array(
            'color' => $color,
            'alpha' => $alpha,
        );
    }
    return $slots;
}

function mdp_hex_to_rgba($hex, $alpha = 1.0) {
    $hex = sanitize_hex_color($hex);
    if (!$hex) {
        $hex = '#000000';
    }
    $hex = ltrim($hex, '#');
    if (strlen($hex) === 3) {
        $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
    }
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    return 'rgba(' . $r . ',' . $g . ',' . $b . ',' . $alpha . ')';
}

function mdp_get_email_schedule_description($interval) {
    $descriptions = array(
        'disabled' => __('Deaktiviert', 'elementor-forms-statistics'),
        'daily' => __('Täglich', 'elementor-forms-statistics'),
        'weekly' => __('Wöchentlich', 'elementor-forms-statistics'),
        'monthly' => __('Monatlich', 'elementor-forms-statistics'),
    );
    return isset($descriptions[$interval]) ? $descriptions[$interval] : $interval;
}

function mdp_get_email_send_time() {
    $value = get_option('mdp_efs_email_time', '08:00');
    if (!is_string($value) || !preg_match('/^([01]\d|2[0-3]):([0-5]\d)$/', $value)) {
        $value = '08:00';
    }
    return $value;
}

function mdp_get_email_day_of_month() {
    $value = (int) get_option('mdp_efs_email_day_of_month', 1);
    if ($value < 1) {
        $value = 1;
    } elseif ($value > 31) {
        $value = 31;
    }
    return $value;
}

function mdp_get_email_weekday_choices() {
    return array(
        'monday' => __('Montag', 'elementor-forms-statistics'),
        'tuesday' => __('Dienstag', 'elementor-forms-statistics'),
        'wednesday' => __('Mittwoch', 'elementor-forms-statistics'),
        'thursday' => __('Donnerstag', 'elementor-forms-statistics'),
        'friday' => __('Freitag', 'elementor-forms-statistics'),
        'saturday' => __('Samstag', 'elementor-forms-statistics'),
        'sunday' => __('Sonntag', 'elementor-forms-statistics'),
    );
}

function mdp_get_email_weekday() {
    $value = strtolower(get_option('mdp_efs_email_weekday', 'monday'));
    $choices = mdp_get_email_weekday_choices();
    if (!isset($choices[$value])) {
        $value = 'monday';
    }
    return $value;
}

function mdp_get_email_time_parts() {
    $time = mdp_get_email_send_time();
    $parts = explode(':', $time);
    $hour = isset($parts[0]) ? (int) $parts[0] : 8;
    $minute = isset($parts[1]) ? (int) $parts[1] : 0;
    return array($hour, $minute);
}

function mdp_calculate_next_email_timestamp($interval) {
    if ($interval === 'disabled') {
        return 0;
    }

    $timezone = wp_timezone();
    $now = new DateTime('now', $timezone);
    list($hour, $minute) = mdp_get_email_time_parts();

    if ($interval === 'daily') {
        $target = clone $now;
        $target->setTime($hour, $minute, 0);
        if ($target <= $now) {
            $target->modify('+1 day');
        }
        return $target->getTimestamp();
    }

    if ($interval === 'weekly') {
        $weekday = mdp_get_email_weekday();
        $target = clone $now;
        $target->setTime($hour, $minute, 0);
        $current_weekday = strtolower($now->format('l'));
        if ($weekday === $current_weekday && $target > $now) {
            return $target->getTimestamp();
        }
        $target = new DateTime('next ' . $weekday, $timezone);
        $target->setTime($hour, $minute, 0);
        return $target->getTimestamp();
    }

    if ($interval === 'monthly') {
        $day = mdp_get_email_day_of_month();
        $target = clone $now;
        $target_day = min($day, (int) $target->format('t'));
        $target->setDate((int) $target->format('Y'), (int) $target->format('m'), $target_day);
        $target->setTime($hour, $minute, 0);
        if ($target <= $now) {
            $target->modify('first day of next month');
            $target_day = min($day, (int) $target->format('t'));
            $target->setDate((int) $target->format('Y'), (int) $target->format('m'), $target_day);
            $target->setTime($hour, $minute, 0);
        }
        return $target->getTimestamp();
    }

    return time() + MINUTE_IN_SECONDS;
}

function mdp_get_cron_schedule_slug($interval) {
    switch ($interval) {
        case 'daily':
            return 'daily';
        case 'weekly':
            return 'weekly';
        case 'monthly':
            return 'mdp_monthly';
        default:
            return '';
    }
}

function mdp_add_custom_cron_schedules($schedules) {
    if (!isset($schedules['mdp_monthly'])) {
        $schedules['mdp_monthly'] = array(
            'interval' => DAY_IN_SECONDS * 30,
            'display'  => __('Einmal im Monat', 'elementor-forms-statistics'),
        );
    }
    return $schedules;
}

function mdp_maybe_schedule_stats_email() {
    $interval = mdp_get_schedule_interval();
    $recipients = mdp_get_email_recipients();
    $hook = 'mdp_send_stats_email';
    $schedule_mode = get_option('mdp_efs_schedule_mode', '');
    if ($schedule_mode !== 'single') {
        wp_clear_scheduled_hook($hook);
        update_option('mdp_efs_schedule_mode', 'single');
    }

    if ($interval === 'disabled' || empty($recipients)) {
        wp_clear_scheduled_hook($hook);
        return;
    }

    if (wp_next_scheduled($hook)) {
        return;
    }

    $next_timestamp = mdp_calculate_next_email_timestamp($interval);
    if ($next_timestamp > 0) {
        wp_schedule_single_event($next_timestamp, $hook);
    }
}

function mdp_reset_stats_email_schedule() {
    wp_clear_scheduled_hook('mdp_send_stats_email');
    mdp_maybe_schedule_stats_email();
}

function mdp_get_submission_cleanup_interval() {
    $interval = get_option('mdp_efs_submission_cleanup_interval', 'disabled');
    if ($interval === 'disabled') {
        return 'disabled';
    }
    if (preg_match('/^(\d+)([hdmy])$/', $interval)) {
        return $interval;
    }
    return 'disabled';
}

function mdp_get_submission_cleanup_enabled() {
    return get_option('mdp_efs_submission_cleanup_enabled', '0') === '1';
}

function mdp_get_submission_cleanup_value() {
    $value = get_option('mdp_efs_submission_cleanup_value', 0);
    return max(0, intval($value));
}

function mdp_get_submission_cleanup_unit() {
    $unit = get_option('mdp_efs_submission_cleanup_unit', 'd');
    $allowed = array('h', 'd', 'm', 'y');
    if (in_array($unit, $allowed, true)) {
        return $unit;
    }
    return 'd';
}

function mdp_get_submission_cleanup_config() {
    $enabled = mdp_get_submission_cleanup_enabled();
    $value = mdp_get_submission_cleanup_value();
    $unit = mdp_get_submission_cleanup_unit();
    $allowed = array('h', 'd', 'm', 'y');
    if ($enabled && $value > 0 && in_array($unit, $allowed, true)) {
        return array(
            'enabled' => true,
            'value' => $value,
            'unit' => $unit,
        );
    }

    $legacy = mdp_get_submission_cleanup_interval();
    if ($legacy !== 'disabled' && preg_match('/^(\d+)([hdmy])$/', $legacy, $matches)) {
        return array(
            'enabled' => true,
            'value' => (int) $matches[1],
            'unit' => $matches[2],
        );
    }

    return array(
        'enabled' => false,
        'value' => 0,
        'unit' => 'h',
    );
}

function mdp_get_submission_cleanup_seconds() {
    $config = mdp_get_submission_cleanup_config();
    if (!$config['enabled']) {
        return 0;
    }
    switch ($config['unit']) {
        case 'h':
            return max(1, $config['value']) * HOUR_IN_SECONDS;
        case 'd':
            return max(1, $config['value']) * DAY_IN_SECONDS;
        case 'm':
            return max(1, $config['value']) * DAY_IN_SECONDS * 30;
        case 'y':
            return max(1, $config['value']) * DAY_IN_SECONDS * 365;
    }
    return 0;
}

function mdp_clear_elementor_submissions_callback() {
    global $wpdb;
    $seconds = mdp_get_submission_cleanup_seconds();
    if ($seconds <= 0) {
        wp_clear_scheduled_hook('mdp_clear_elementor_submissions');
        return;
    }
    $threshold = date('Y-m-d H:i:s', current_time('timestamp') - $seconds);
    $status_like = '%trash%';
    $condition = $wpdb->prepare('s.status NOT LIKE %s AND s.created_at_gmt <= %s', $status_like, $threshold);
    $values_table = $wpdb->prefix . 'e_submissions_values';
    $actions_log_table = $wpdb->prefix . 'e_submissions_actions_log';
    $submissions_table = $wpdb->prefix . 'e_submissions';
    $wpdb->query("DELETE ev FROM {$values_table} ev INNER JOIN {$submissions_table} s ON ev.submission_id = s.ID WHERE {$condition}");
    $wpdb->query("DELETE log FROM {$actions_log_table} log INNER JOIN {$submissions_table} s ON log.submission_id = s.ID WHERE {$condition}");
    $wpdb->query("DELETE FROM {$submissions_table} s WHERE {$condition}");
    mdp_schedule_submission_cleanup(true);
}

function mdp_schedule_submission_cleanup($force_reschedule = false) {
    $seconds = mdp_get_submission_cleanup_seconds();
    if ($seconds <= 0) {
        wp_clear_scheduled_hook('mdp_clear_elementor_submissions');
        return;
    }
    if ($force_reschedule) {
        wp_clear_scheduled_hook('mdp_clear_elementor_submissions');
    }
    if (!wp_next_scheduled('mdp_clear_elementor_submissions')) {
        wp_schedule_single_event(time() + $seconds, 'mdp_clear_elementor_submissions');
    }
}

add_action('mdp_clear_elementor_submissions', 'mdp_clear_elementor_submissions_callback');

function mdp_send_stats_email_callback($manual = false) {
    $recipients = mdp_get_email_recipients();
    if (empty($recipients)) {
        if (!$manual) {
            wp_clear_scheduled_hook('mdp_send_stats_email');
        }
        return false;
    }

    $include_attachment = mdp_should_include_stats_attachment();

    $subject = mdp_build_email_subject(date_i18n('F Y'));
    $message = mdp_get_email_message();
    $site_url_text = esc_url(trailingslashit(home_url()));
    $email_font_stack = "-apple-system, BlinkMacSystemFont, 'SF Pro Display', 'SF Pro Text', 'Segoe UI', 'Open Sans', 'Helvetica Neue', sans-serif";
    $blog_name = wp_specialchars_decode(get_option('blogname'), ENT_QUOTES);
    $headers = array(
        'Content-Type: text/html; charset=' . get_option('blog_charset'),
        'From: ' . $blog_name . ' <' . get_option('admin_email') . '>',
    );

    $html_export = custom_menu_callback(true, array(
        'export' => true,
        'inline_styles' => true,
        'include_html_document' => true,
        'include_chartjs' => true,
    ));

    $export_file = mdp_save_stats_export_file($html_export);
    $download_url = '';
    if (!empty($export_file['path'])) {
        $download_url = mdp_get_stats_export_download_url(mdp_create_stats_export_token($export_file['path']));
    }
    $link_note = '';
    $link_label = mdp_get_export_link_label();
    $button_html = mdp_build_stats_link_button($download_url, $link_label);
    $replacements = array(
        '%s' => $site_url_text,
        '%stats_link%' => $button_html,
        'stats_link%' => $button_html,
        '%stats_link' => $button_html,
    );
    $message_with_link = strtr($message, $replacements);
    if ($export_url) {
        $message_with_link = preg_replace(
            '/https?:\\/\\/[\\w\\-\\.\\/%]+stats_link%/i',
            $button_html,
            $message_with_link
        );
    }

    $body = mdp_build_styled_email_body($message_with_link, $link_note, $export_url, $link_label);

    $attachments = array();
    if ($include_attachment) {
        $temp_dir = trailingslashit(get_temp_dir());
        $attachment_filename = 'Anfragen Statistik ' . date_i18n('Y-m') . '.html';
        $attachment_basename = wp_unique_filename($temp_dir, $attachment_filename);
        $attachment_path = $temp_dir . $attachment_basename;
        if (file_put_contents($attachment_path, $html_export) !== false) {
            $attachments[] = $attachment_path;
        }
    }

    $sent = wp_mail($recipients, $subject, $body, $headers, $attachments);

    if (!empty($attachments)) {
        foreach ($attachments as $attachment) {
            if (file_exists($attachment)) {
                unlink($attachment);
            }
        }
    }

    if (!$manual) {
        wp_clear_scheduled_hook('mdp_send_stats_email');
        mdp_maybe_schedule_stats_email();
    }

    return (bool) $sent;
}

function mdp_get_stats_export_folder_slug() {
    $option = get_option('mdp_efs_export_folder', 'mdpstats');
    $slug = sanitize_title($option);
    if ($slug === '') {
        $slug = 'elementor-forms-statistics';
    }
    return $slug;
}

function mdp_get_stats_export_subdir() {
    return mdp_get_stats_export_folder_slug();
}

function mdp_get_stats_export_dir() {
    $uploads = wp_upload_dir();
    return trailingslashit($uploads['basedir']) . mdp_get_stats_export_subdir();
}

function mdp_get_stats_export_url() {
    $uploads = wp_upload_dir();
    return trailingslashit($uploads['baseurl']) . mdp_get_stats_export_subdir();
}

function mdp_ensure_stats_export_dir() {
    $dir = mdp_get_stats_export_dir();
    if (!wp_mkdir_p($dir)) {
        return false;
    }
    mdp_ensure_stats_export_htaccess($dir);
    return $dir;
}

function mdp_ensure_stats_export_htaccess($dir) {
    $htaccess_path = trailingslashit($dir) . '.htaccess';
    $lines = array(
        '# Generated by MDP Elementor Forms Extras',
        '# Der Ordner ist nur über Token-Links erreichbar.',
        '<IfModule mod_headers.c>',
        'Header set X-Robots-Tag "noindex, nofollow, noarchive"',
        '</IfModule>',
        'Options -Indexes',
    );
    file_put_contents($htaccess_path, implode("\n", $lines) . "\n", LOCK_EX);
}

function mdp_clear_stats_export_folder_contents($dir = '') {
    if ($dir === '') {
        $dir = mdp_get_stats_export_dir();
    }
    if (!$dir || !is_dir($dir)) {
        return;
    }
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..' || $item === '.htaccess') {
            continue;
        }
        $path = trailingslashit($dir) . $item;
        if (is_dir($path)) {
            mdp_remove_directory_tree($path);
        } else {
            @unlink($path);
        }
    }
}

function mdp_remove_directory_tree($dir) {
    if (!$dir || !is_dir($dir)) {
        return;
    }
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') {
            continue;
        }
        $path = trailingslashit($dir) . $item;
        if (is_dir($path)) {
            mdp_remove_directory_tree($path);
        } else {
            @unlink($path);
        }
    }
    @rmdir($dir);
}

function mdp_build_stats_link_button($download_url, $link_label) {
    if (!$download_url) {
        return '';
    }
    $label = $link_label !== '' ? $link_label : __('Statistik öffnen', 'elementor-forms-statistics');
    $url = esc_url($download_url);
    return '<a href="' . $url . '" target="_blank" rel="noopener noreferrer" style="display:inline-block;margin:16px auto 0;padding:14px 32px;background:#1f2a44;color:#ffffff;font-weight:600;text-decoration:none;border-radius:999px;box-shadow:0 10px 30px rgba(31,42,68,0.25);">' . esc_html($label) . '</a>';
}

function mdp_uninstall_plugin() {
    if (get_option('mdp_efs_clean_on_uninstall', '0') !== '1') {
        return;
    }

    global $wpdb;
    wp_clear_scheduled_hook('mdp_send_stats_email');
    wp_clear_scheduled_hook('mdp_clear_elementor_submissions');
    $table = mdp_get_archive_table_name();
    $wpdb->query("DROP TABLE IF EXISTS {$table}");

    $like_prefix = $wpdb->esc_like('mdp_efs_') . '%';
    $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", $like_prefix));

    mdp_remove_directory_tree(mdp_get_stats_export_dir());
}

function mdp_save_stats_export_file($html_export) {
    $dir = mdp_ensure_stats_export_dir();
    if (!$dir || $html_export === '') {
        return '';
    }
    mdp_clear_stats_export_folder_contents($dir);
    $timestamp = date_i18n('Y-m-d_H-i-s');
    $random = substr(wp_generate_password(8, false, false), 0, 8);
    $base_name = 'anfragen-statistik-' . $timestamp . '-' . $random;
    $filename = wp_unique_filename($dir, sanitize_file_name($base_name . '.html'));
    $path = trailingslashit($dir) . $filename;
    if (file_put_contents($path, $html_export) === false) {
        return '';
    }
    return array(
        'path' => $path,
        'url' => trailingslashit(mdp_get_stats_export_url()) . rawurlencode($filename),
    );
}

function mdp_get_stats_link_placeholder() {
    return '%stats_link%';
}

function mdp_get_export_link_label() {
    $default = __('Statistik öffnen', 'elementor-forms-statistics');
    $label = get_option('mdp_efs_export_link_label', $default);
    if (!is_string($label) || $label === '') {
        return $default;
    }
    return $label;
}

function mdp_get_stats_export_token_ttl() {
    $interval = mdp_get_schedule_interval();
    switch ($interval) {
        case 'daily':
            return DAY_IN_SECONDS;
        case 'weekly':
            return WEEK_IN_SECONDS;
        case 'monthly':
            return DAY_IN_SECONDS * 30;
        default:
            return DAY_IN_SECONDS * 30;
    }
}

function mdp_create_stats_export_token($file_path) {
    if ($file_path === '' || !file_exists($file_path)) {
        return '';
    }
    $token = wp_generate_password(24, false, false);
    $expires = time() + mdp_get_stats_export_token_ttl();
    $record = array(
        'token' => $token,
        'path' => $file_path,
        'expires' => $expires,
    );
    update_option('mdp_efs_export_token', $record);
    return $token;
}

function mdp_get_stats_export_download_url($token) {
    if ($token === '') {
        return '';
    }
    return add_query_arg(
        array(
            'mdp_stats_export' => $token,
        ),
        home_url('/')
    );
}

function mdp_handle_stats_export_request() {
    $token = get_query_var('mdp_stats_export', '');
    if ($token === '') {
        return;
    }
    $record = get_option('mdp_efs_export_token', array());
    if (empty($record['token']) || !hash_equals($record['token'], $token)) {
        wp_die(__('Ungültiger Token.', 'elementor-forms-statistics'));
    }
    $expires = isset($record['expires']) ? (int) $record['expires'] : 0;
    if ($expires && time() > $expires) {
        wp_die(__('Dieser Link ist abgelaufen.', 'elementor-forms-statistics'));
    }
    $path = isset($record['path']) ? $record['path'] : '';
    if ($path === '' || !file_exists($path)) {
        wp_die(__('Die Datei wurde nicht gefunden.', 'elementor-forms-statistics'));
    }

    nocache_headers();
    header('Content-Description: File Transfer');
    header('Content-Type: text/html; charset=' . get_option('blog_charset'));
    header('Content-Disposition: inline; filename="anfragen-statistik-' . date_i18n('Y-m-d') . '.html"');
    header('Content-Length: ' . filesize($path));
    readfile($path);
    exit;
}

add_action('init', function () {
    add_filter('query_vars', function ($vars) {
        $vars[] = 'mdp_stats_export';
        return $vars;
    });
});

add_action('template_redirect', 'mdp_handle_stats_export_request');

function mdp_send_stats_now_handler() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('Sie haben keine Berechtigung, diese Seite zu sehen.', 'elementor-forms-statistics'));
    }

    if (!isset($_POST['mdp_send_now_nonce']) || !wp_verify_nonce($_POST['mdp_send_now_nonce'], 'mdp_send_stats_now')) {
        wp_die(__('Ungültige Anfrage.', 'elementor-forms-statistics'));
    }

    $result = mdp_send_stats_email_callback(true);
    $status = $result ? 'success' : 'error';
    $redirect = wp_get_referer();
    if (!$redirect) {
        $redirect = admin_url('admin.php?page=statistiken-einstellungen');
    }
    wp_safe_redirect(add_query_arg('mdp_send_status', $status, $redirect));
    exit;
}

function mdp_get_font_base64($absolute_path) {
    if (!file_exists($absolute_path)) {
        return '';
    }
    $contents = file_get_contents($absolute_path);
    if ($contents === false) {
        return '';
    }
    return base64_encode($contents);
}

function mdp_get_inline_export_styles() {
    $css = '';
    $css_path = plugin_dir_path(__FILE__) . 'assets/css/style.css';
    if (file_exists($css_path)) {
        $file_css = file_get_contents($css_path);
        if ($file_css !== false) {
            $css .= $file_css;
        }
    }

    $dashicons_path = trailingslashit(ABSPATH) . 'wp-includes/fonts/dashicons.woff2';
    $dashicons_base64 = mdp_get_font_base64($dashicons_path);
    if ($dashicons_base64) {
        $css .= "@font-face{font-family:'dashicons';font-style:normal;font-weight:normal;font-display:swap;src:url(data:font/woff2;base64," . $dashicons_base64 . ") format('woff2');}";
    }

    $font_stack = "-apple-system, BlinkMacSystemFont, 'SF Pro Display', 'SF Pro Text', 'Segoe UI', 'Open Sans', 'Helvetica Neue', sans-serif";
    $css .= "
    body.mdp-export-body,
    .mdp-email-container {
        font-family: {$font_stack};
        background: #e7eaef;
        color: #1d2327;
        margin: 0;
        padding: 32px 0 48px;
    }
    .mdp-email-container {
        width: 100%;
    }
    body.mdp-export-body .wrap,
    .mdp-email-container .wrap {
        max-width: 1180px;
        margin: 0 auto;
        background: transparent;
        padding: 0 40px;
        border-radius: 0;
        box-shadow: none;
        border: 0;
        font-family: {$font_stack};
    }
    .mdp-email-message {
        font-family: {$font_stack};
        max-width: 1180px;
        margin: 0 auto 24px;
        padding: 0 40px;
        color: #1d2327;
        line-height: 1.5;
        font-size: 14px;
    }
    .mdp-email-container .wrap.mdp-stats-root,
    .mdp-email-container h1,
    .mdp-email-container h2,
    .mdp-email-container .mdp-version,
    .mdp-email-container table,
    .mdp-email-container .mdp-chart,
    .mdp-email-container .mdp-table,
    .mdp-email-container .checkbox-container,
    .mdp-email-container p {
        font-family: {$font_stack};
    }
    body.mdp-export-body .mdp-chart,
    body.mdp-export-body .mdp-table,
    .mdp-email-container .mdp-chart,
    .mdp-email-container .mdp-table {
        width: 100%;
        max-width: 1090px;
        margin-left: auto;
        margin-right: auto;
    }
    .mdp-email-container .mdp-year-heading,
    body.mdp-export-body .mdp-year-heading {
        max-width: 1090px;
        margin: 40px auto 10px;
        font-size: 18px;
        font-weight: 600;
        color: #1d2327;
    }
    body.mdp-export-body .checkbox-container,
    .mdp-email-container .checkbox-container {
        grid-template-columns: repeat(auto-fit, minmax(220px, 1fr));
    }
    body.mdp-export-body .mdp-chart,
    .mdp-email-container .mdp-chart {
        height: 520px;
    }
    body.mdp-export-body .mdp-version,
    .mdp-email-container .mdp-version {
        text-align: center;
        color: #50575e;
    }
    body.mdp-export-body h1,
    body.mdp-export-body h2,
    .mdp-email-container h1,
    .mdp-email-container h2 {
        color: #1d2327;
    }
    ";

    return $css;
}

function mdp_render_admin_section_styles() {
    ?>
    <style>
    .mdp-settings-wrapper {
        max-width: 1080px;
        margin: 0;
    }
    .mdp-color-table {
        width: auto;
    }
    .mdp-color-table .mdp-color-label {
        width: 90px;
        padding-right: 8px;
    }
    .mdp-color-table .mdp-color-input-cell {
        width: 90px;
    }
    .mdp-settings-wrapper .form-table {
        margin-left: 0;
    }
    .mdp-settings-wrapper .form-table th {
        padding-left: 0;
        padding-right: 24px;
    }
    .mdp-settings-wrapper .form-table td {
        padding-left: 0;
    }
    </style>
    <?php
}

function mdp_get_emphasis_threshold() {
    $value = get_option('mdp_efs_emphasis_threshold', 5);
    $value = is_numeric($value) ? floatval($value) : 5;
    if ($value < 0) {
        $value = 0;
    }
    return $value;
}

function mdp_get_default_submission_access_roles() {
    return array('editor');
}

function mdp_get_submission_access_roles() {
    $raw = get_option('mdp_efs_submission_access_roles', false);
    $has_option = $raw !== false;
    $roles = $has_option ? $raw : mdp_get_default_submission_access_roles();
    if (!is_array($roles)) {
        $roles = array();
    }
    $clean = array();
    foreach ($roles as $role) {
        if (!is_string($role)) {
            continue;
        }
        $role = sanitize_text_field($role);
        if ($role === '') {
            continue;
        }
        $clean[] = $role;
    }
    $clean = array_values(array_unique($clean));
    if (empty($clean) && !$has_option) {
        $clean = mdp_get_default_submission_access_roles();
    }
    return $clean;
}

function mdp_user_has_submission_access_role($user_id = 0) {
    if (empty($user_id)) {
        $user_id = get_current_user_id();
    }
    if (empty($user_id)) {
        return false;
    }
    $user = get_userdata($user_id);
    if (!$user) {
        return false;
    }
    $allowed_roles = mdp_get_submission_access_roles();
    foreach ((array) $user->roles as $role) {
        if (in_array($role, $allowed_roles, true)) {
            return true;
        }
    }
    return false;
}

/**
 * Calculate an inline background style comparing current month values with previous year.
 */
function mdp_get_comparison_background($current_value, $previous_value, $has_previous_year_data) {
    if (!$has_previous_year_data) {
        return '';
    }

    $current_value = (int) $current_value;
    $previous_value = (int) $previous_value;

    if ($current_value === $previous_value) {
        return '';
    }

    $delta = $current_value - $previous_value;
    $threshold = mdp_get_emphasis_threshold();
    if ($threshold > 0 && abs($delta) < $threshold) {
        return '';
    }
    $ratio = 0.0;

    if ($previous_value === 0) {
        if ($current_value === 0) {
            return '';
        }
        $ratio = min($current_value / 5, 1); // normalize growth when no previous data
    } else {
        $ratio = min(abs($delta) / max($previous_value, 1), 1);
    }

    $alpha = 0.03 + (0.12 * $ratio); // subtle pastel intensity
    $color = $delta > 0
        ? 'rgba(76, 175, 80,' . $alpha . ')'   // green tones
        : 'rgba(244, 67, 54,' . $alpha . ')';  // red tones

    return ' style="background-color:' . esc_attr($color) . ';"';
}

function mdp_should_emphasize_change($current_value, $previous_value, $has_previous_year_data) {
    if (!$has_previous_year_data) {
        return false;
    }

    $current_value = (int) $current_value;
    $previous_value = (int) $previous_value;

    if ($current_value === $previous_value) {
        return false;
    }

    $delta = abs($current_value - $previous_value);
    $threshold = mdp_get_emphasis_threshold();
    return $delta >= $threshold;
}

function mdp_get_excluded_emails() {
    $raw = get_option('mdp_efs_excluded_emails', '');
    if (!is_string($raw) || $raw === '') {
        return [];
    }
    $lines = preg_split('/\r\n|\r|\n/', $raw);
    $patterns = [];
    foreach ($lines as $line) {
        $trimmed = trim($line);
        if ($trimmed === '') {
            continue;
        }
        $patterns[] = strtolower(sanitize_text_field($trimmed));
    }
    return array_values(array_unique(array_filter($patterns)));
}

function mdp_get_excluded_emails_text() {
    $emails = mdp_get_excluded_emails();
    return implode("\n", $emails);
}

function mdp_prepare_email_pattern_for_like($pattern) {
    $pattern = strtolower(trim($pattern));
    if ($pattern === '') {
        return '';
    }
    $pattern = str_replace(array('*', '?'), array('%', '_'), $pattern);
    if (strpos($pattern, '%') === false && strpos($pattern, '_') === false) {
        $pattern = '%' . $pattern . '%';
    }
    return $pattern;
}

function mdp_get_email_exclusion_clause($table_alias) {
    $emails = mdp_get_excluded_emails();
    if (empty($emails)) {
        return '';
    }

    $like_conditions = [];
    $params = array('%email%');

    foreach ($emails as $pattern) {
        $like_pattern = mdp_prepare_email_pattern_for_like($pattern);
        if ($like_pattern === '') {
            continue;
        }
        $like_conditions[] = 'LOWER(ev.value) LIKE %s';
        $params[] = $like_pattern;
    }

    if (empty($like_conditions)) {
        return '';
    }

    global $wpdb;
    $condition_sql = implode(' OR ', $like_conditions);

    return $wpdb->prepare(
        " AND NOT EXISTS (
            SELECT 1 FROM " . $wpdb->prefix . "e_submissions_values ev
            WHERE ev.submission_id = {$table_alias}.ID
            AND LOWER(ev.`key`) LIKE %s
            AND ({$condition_sql})
        )",
        $params
    );
}

function mdp_settings_page_callback() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('Sie haben keine Berechtigung, diese Seite zu sehen.', 'elementor-forms-statistics'));
    }

    if (!function_exists('wp_roles')) {
        require_once ABSPATH . 'wp-includes/capabilities.php';
    }
    $wp_roles = wp_roles();
    $role_objects = $wp_roles ? $wp_roles->roles : array();
    $role_keys = array_keys($role_objects);

    $message = '';
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mdp_settings_nonce']) && wp_verify_nonce($_POST['mdp_settings_nonce'], 'mdp_save_settings')) {
        $emails_raw = isset($_POST['excluded_emails']) ? wp_unslash($_POST['excluded_emails']) : '';
        $lines = preg_split('/\r\n|\r|\n/', $emails_raw);
        $sanitized = [];
        foreach ($lines as $line) {
            $trimmed = trim($line);
            if ($trimmed === '') {
                continue;
            }
            $pattern = sanitize_text_field($trimmed);
            if ($pattern !== '') {
                $sanitized[] = strtolower($pattern);
            }
        }
        $submitted_roles = isset($_POST['submission_access_roles']) && is_array($_POST['submission_access_roles']) ? $_POST['submission_access_roles'] : array();
        $selected_roles = array();
        foreach ($submitted_roles as $role_input) {
            $role_input = sanitize_text_field($role_input);
            if ($role_input === '' || !in_array($role_input, $role_keys, true)) {
                continue;
            }
            $selected_roles[] = $role_input;
        }
        $selected_roles = array_values(array_unique($selected_roles));
        $threshold_value = isset($_POST['emphasis_threshold']) ? floatval($_POST['emphasis_threshold']) : 5;
        if ($threshold_value < 0) {
            $threshold_value = 0;
        }
        $export_folder_slug = isset($_POST['export_folder']) ? sanitize_title($_POST['export_folder']) : '';
        if ($export_folder_slug === '') {
            $export_folder_slug = 'mdpstats';
        }
        $clean_on_uninstall = isset($_POST['clean_on_uninstall']) ? '1' : '0';
        $cleanup_enabled = isset($_POST['submission_cleanup_enabled']) ? '1' : '0';
        $cleanup_value = isset($_POST['submission_cleanup_value']) ? intval($_POST['submission_cleanup_value']) : 0;
        $cleanup_unit = isset($_POST['submission_cleanup_unit']) ? sanitize_text_field($_POST['submission_cleanup_unit']) : 'd';
        $allowed_units = array('h', 'd', 'm', 'y');
        if (!in_array($cleanup_unit, $allowed_units, true)) {
            $cleanup_unit = 'd';
        }
        if ($cleanup_value < 1) {
            $cleanup_enabled = '0';
            $cleanup_value = 0;
        }
        $legacy_interval = 'disabled';
        if ($cleanup_enabled === '1' && $cleanup_value > 0) {
            $legacy_interval = $cleanup_value . $cleanup_unit;
        }

        $curve_color_slots = array();
        $default_curve_slots = mdp_get_default_curve_color_slots();
        for ($i = 0; $i < 10; $i++) {
            $color_field = isset($_POST['curve_color_' . $i]) ? sanitize_hex_color($_POST['curve_color_' . $i]) : '';
            $alpha_field = isset($_POST['curve_alpha_' . $i]) ? floatval($_POST['curve_alpha_' . $i]) : round($default_curve_slots[$i]['alpha'] * 100);
            $alpha_value = max(0, min(100, $alpha_field)) / 100;
            if ($color_field === '') {
                $color_field = $default_curve_slots[$i]['color'];
            }
            $curve_color_slots[] = array(
                'color' => $color_field,
                'alpha' => $alpha_value,
            );
        }

        update_option('mdp_efs_selected_form_ids', isset($_POST['selected_form_ids']) && is_array($_POST['selected_form_ids']) ? array_map('sanitize_text_field', $_POST['selected_form_ids']) : []);
        update_option('mdp_efs_submission_access_roles', $selected_roles);
        update_option('mdp_efs_excluded_emails', implode("\n", $sanitized));
        update_option('mdp_efs_emphasis_threshold', $threshold_value);
        update_option('mdp_efs_export_folder', $export_folder_slug);
        update_option('mdp_efs_curve_color_slots', $curve_color_slots);
        update_option('mdp_efs_clean_on_uninstall', $clean_on_uninstall);
        update_option('mdp_efs_submission_cleanup_interval', $legacy_interval);
        update_option('mdp_efs_submission_cleanup_enabled', $cleanup_enabled);
        update_option('mdp_efs_submission_cleanup_value', $cleanup_value);
        update_option('mdp_efs_submission_cleanup_unit', $cleanup_unit);
        mdp_ensure_stats_export_dir();
        $message = __('Einstellungen gespeichert.', 'elementor-forms-statistics');
        mdp_schedule_submission_cleanup(true);
    }

    $stored_text = get_option('mdp_efs_excluded_emails', '');
    $forms_indexed = mdp_get_forms_indexed();
    $selected_form_ids = mdp_get_selected_form_ids();
    $emphasis_threshold = mdp_get_emphasis_threshold();
    $export_folder_slug = mdp_get_stats_export_folder_slug();
    $curve_color_slots = mdp_get_curve_color_slots();
    $clean_on_uninstall = get_option('mdp_efs_clean_on_uninstall', '0');
    $cleanup_config = mdp_get_submission_cleanup_config();
    $cleanup_enabled = $cleanup_config['enabled'];
    $cleanup_value = $cleanup_config['value'];
    $cleanup_unit = $cleanup_config['unit'];
    $cleanup_unit_choices = array(
        'h' => __('Stunden', 'elementor-forms-statistics'),
        'd' => __('Tage', 'elementor-forms-statistics'),
        'm' => __('Monate', 'elementor-forms-statistics'),
        'y' => __('Jahre', 'elementor-forms-statistics'),
    );
    $submission_access_roles = mdp_get_submission_access_roles();
    ?>
    <div class="wrap">
        <?php mdp_render_admin_section_styles(); ?>
        <div class="mdp-settings-wrapper">
        <h1><?php _e('Einstellungen', 'elementor-forms-statistics'); ?></h1>
        <?php if ($message) : ?>
            <div class="notice notice-success"><p><?php echo esc_html($message); ?></p></div>
        <?php endif; ?>
        <form method="post">
            <?php wp_nonce_field('mdp_save_settings', 'mdp_settings_nonce'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label><?php _e('Sichtbare Formulare in Statistik', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <?php if (empty($forms_indexed)) : ?>
                            <p><?php _e('Keine Seiten gefunden.', 'elementor-forms-statistics'); ?></p>
                        <?php else : ?>
                            <div class="checkbox-container">
                                <?php foreach ($forms_indexed as $form_id => $form_entry) :
                                    $resolved_title = mdp_resolve_form_title($form_id, $form_entry);
                                    $label_title = $resolved_title !== '' ? $resolved_title : __('Unbenannte Seite', 'elementor-forms-statistics');
                                    $checked = in_array($form_id, $selected_form_ids, true) ? 'checked' : '';
                                ?>
                                    <label>
                                        <input type="checkbox" name="selected_form_ids[]" value="<?php echo esc_attr($form_id); ?>" <?php echo $checked; ?>>
                                        <span class="checkbox-label-text" title="<?php echo esc_attr($label_title); ?>"><?php echo esc_html($label_title); ?></span>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                            <p class="description"><?php _e('Nur ausgewählte Formulare werden in Grafik und Tabellen berücksichtigt.', 'elementor-forms-statistics'); ?></p>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label><?php _e('Benutzergruppen mit Formularzugriff', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <?php if (empty($role_objects)) : ?>
                            <p><?php _e('Keine Benutzergruppen gefunden.', 'elementor-forms-statistics'); ?></p>
                        <?php else : ?>
                            <div class="checkbox-container">
                                <?php foreach ($role_objects as $role_key => $role_data) :
                                    $checked = in_array($role_key, $submission_access_roles, true);
                                ?>
                                    <label>
                                        <input type="checkbox" name="submission_access_roles[]" value="<?php echo esc_attr($role_key); ?>" <?php checked($checked); ?>>
                                        <span class="checkbox-label-text" title="<?php echo esc_attr($role_data['name']); ?>"><?php echo esc_html($role_data['name']); ?></span>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                        <p class="description"><?php _e('Diese Gruppen erhalten zusätzlich zum Administratoren-Backend Zugriff auf Elementor-Submissions.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="excluded_emails"><?php _e('E-Mail Ignorelist', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <textarea name="excluded_emails" id="excluded_emails" rows="8" class="large-text code"><?php echo esc_textarea($stored_text); ?></textarea>
                        <p class="description"><?php _e('E-Mails in diesem Feld fließen nicht in die Statistik ein. Bitte je Zeile eine Adresse oder einen Teilstring eintragen; Wildcards wie * können verwendet werden.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="emphasis_threshold"><?php _e('Schwellwert für außergewöhnliche Veränderungen', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <input type="number" min="0" step="1" name="emphasis_threshold" id="emphasis_threshold" value="<?php echo esc_attr($emphasis_threshold); ?>">
                        <p class="description"><?php _e('Ab dieser Differenz zum Vorjahr werden Werte in der Tabelle fett hervorgehoben. Standard: 5.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="export_folder"><?php _e('Export-Ordner (Upload-Verzeichnis)', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <input type="text" name="export_folder" id="export_folder" value="<?php echo esc_attr($export_folder_slug); ?>" class="regular-text">
                        <p class="description"><?php _e('Ordnername unter wp-content/uploads für die gespeicherten Statistiken.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label><?php _e('Farben', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <div class="mdp-color-matrix">
                            <table class="mdp-color-table">
                                <tbody>
                                <?php foreach ($curve_color_slots as $index => $slot) : ?>
                                    <tr>
                                        <td class="mdp-color-label">
                                            <?php
                                            if ($index === 0) {
                                                echo esc_html__('Aktuelles Jahr', 'elementor-forms-statistics');
                                            } else {
                                                printf(esc_html__('%d Jahr', 'elementor-forms-statistics'), $index + 1);
                                            }
                                            ?>
                                        </td>
                                        <td class="mdp-color-input-cell">
                                            <input type="color" name="curve_color_<?php echo $index; ?>" value="<?php echo esc_attr($slot['color']); ?>" class="mdp-curve-color-input">
                                        </td>
                                        <td class="mdp-transparency-cell">
                                            <label class="mdp-transparency-label">
                                                <?php _e('Transparenz', 'elementor-forms-statistics'); ?>:
                                                <span class="mdp-transparency-value"><?php echo esc_html(round($slot['alpha'] * 100)); ?>%</span>
                                            </label>
                                            <input type="range" min="0" max="100" step="5" name="curve_alpha_<?php echo $index; ?>" value="<?php echo esc_attr(round($slot['alpha'] * 100)); ?>" class="mdp-transparency-slider">
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="clean_on_uninstall"><?php _e('Verhalten Deinstallieren', 'elementor-forms-statistics'); ?></label>
                    </th>
                <td>
                    <label>
                        <input type="checkbox" name="clean_on_uninstall" id="clean_on_uninstall" value="1" <?php checked($clean_on_uninstall, '1'); ?>>
                        <span class="mdp-clean-action-text">⚠️ <?php _e('Beim Deinstallieren dieses Plugins alle Daten bereinigen.', 'elementor-forms-statistics'); ?></span>
                    </label>
                    <p class="description mdp-warning-text"><?php _e('Wenn diese Checkbox aktiviert wird, wird beim Löschen dieses Plugins über WordPress die Tabellen dieses Plugins gelöscht, archivierte Statistiken und die statischen Dateien im Upload-Ordner werden gelöscht.', 'elementor-forms-statistics'); ?></p>
                    <p class="description"><?php _e('Elementor Submissions sind von dieser Einstellung nicht betroffen.', 'elementor-forms-statistics'); ?></p>
                </td>
            </tr>
            <tr>
                <th scope="row">
                    <label for="submission_cleanup_enabled"><?php _e('Datenschutz', 'elementor-forms-statistics'); ?></label>
                </th>
                <td>
                    <label>
                        <input type="checkbox" id="submission_cleanup_enabled" name="submission_cleanup_enabled" value="1" <?php checked($cleanup_enabled, true); ?>>
                        <span style="color:#d63638;"><?php _e('Elementor Submissions nach einem bestimmten Zeitraum automatisch löschen', 'elementor-forms-statistics'); ?></span>
                    </label>
                    <div class="mdp-cleanup-duration"<?php echo $cleanup_enabled ? ' style="margin-top:12px; margin-bottom:8px;"' : ' style="display:none; margin-top:12px; margin-bottom:8px;"'; ?>>
                        <input type="number" min="1" name="submission_cleanup_value" id="submission_cleanup_value" value="<?php echo esc_attr($cleanup_value > 0 ? $cleanup_value : 1); ?>" class="small-text" style="width:80px; margin-right:8px;">
                        <select name="submission_cleanup_unit" id="submission_cleanup_unit">
                            <?php foreach ($cleanup_unit_choices as $unit_key => $unit_label) : ?>
                                <option value="<?php echo esc_attr($unit_key); ?>" <?php selected($cleanup_unit, $unit_key); ?>><?php echo esc_html($unit_label); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <span class="description" style="margin-left:8px;"><?php _e('Nach dieser Dauer werden die Einträge gelöscht.', 'elementor-forms-statistics'); ?></span>
                    </div>
                    <p class="description mdp-cleanup-warning">
                        ⚠️ <?php _e('Wenn aktiviert, werden die Elementor Submissions nach Ablauf der gewählten Dauer automatisch gelöscht. Hinweis: Dieser Vorgang lässt sich nicht rückgängig machen.', 'elementor-forms-statistics'); ?>
                    </p>
                    <p class="description">
                        <?php _e('Diese Funktion kann helfen, Datenschutzbestimmungen einzuhalten. Passen Sie den Zeitraum an Ihre Datenschutzerklärung an.', 'elementor-forms-statistics'); ?>
                    </p>
                </td>
            </tr>
            </table>
                <?php submit_button(__('Speichern', 'elementor-forms-statistics')); ?>
        </form>
        </div>
        <script>
        (function($){
            function mdpUpdateTransparencyValue(input) {
                var $input = $(input);
                var $value = $input.closest('.mdp-transparency-cell').find('.mdp-transparency-value');
                if ($value.length) {
                    $value.text($input.val() + '%');
                }
            }
            $(document).on('input', '.mdp-transparency-slider', function() {
                mdpUpdateTransparencyValue(this);
            });
            $(document).ready(function() {
                $('.mdp-transparency-slider').each(function() {
                    mdpUpdateTransparencyValue(this);
                });
            });
        })(jQuery);
        </script>
        <script>
        (function($){
            function mdpToggleCleanupDuration() {
                var enabled = $('#submission_cleanup_enabled').is(':checked');
                $('.mdp-cleanup-duration').toggle(enabled);
            }
            $(function() {
                $('#submission_cleanup_enabled').on('change', mdpToggleCleanupDuration);
                mdpToggleCleanupDuration();
            });
        })(jQuery);
        </script>
    </div>
    <?php
}

function mdp_archive_page_callback() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('Sie haben keine Berechtigung, diese Seite zu sehen.', 'elementor-forms-statistics'));
    }

    $status = isset($_GET['mdp_archive_status']) ? sanitize_text_field($_GET['mdp_archive_status']) : '';
    $table_exists = mdp_archive_table_exists();
    $initialized = (bool) get_option('mdp_efs_archive_initialized');
    $last_run_raw = get_option('mdp_efs_archive_last_run', '');
    $last_sync_timestamp = (int) get_option('mdp_efs_archive_last_sync', 0);
    $last_run_display = $last_run_raw ? date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($last_run_raw)) : __('Noch kein Import durchgeführt.', 'elementor-forms-statistics');
    $last_sync_display = $last_sync_timestamp ? sprintf(__('%s (vor %s)', 'elementor-forms-statistics'), date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $last_sync_timestamp), human_time_diff($last_sync_timestamp, current_time('timestamp'))) : __('Noch nie', 'elementor-forms-statistics');

    $row_count = 0;
    $distinct_forms = 0;
    $total_requests = 0;
    if ($table_exists) {
        global $wpdb;
        $table = mdp_get_archive_table_name();
        $row_count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
        $distinct_forms = (int) $wpdb->get_var("SELECT COUNT(DISTINCT form_id) FROM {$table}");
        $total_requests = (int) $wpdb->get_var("SELECT SUM(total) FROM {$table}");
    }

    $button_label = $initialized
        ? __('Neue Einträge sichern', 'elementor-forms-statistics')
        : __('Archiv initialisieren', 'elementor-forms-statistics');
    ?>
    <div class="wrap">
        <?php mdp_render_admin_section_styles(); ?>
        <div class="mdp-settings-wrapper">
        <h1><?php _e('Status', 'elementor-forms-statistics'); ?></h1>
        <?php if ($status === 'initialized') : ?>
            <div class="notice notice-success"><p><?php _e('Das Archiv wurde vollständig aufgebaut.', 'elementor-forms-statistics'); ?></p></div>
        <?php elseif ($status === 'synced') : ?>
            <div class="notice notice-success"><p><?php _e('Neue Einträge wurden ins Archiv übernommen.', 'elementor-forms-statistics'); ?></p></div>
        <?php endif; ?>
        <p><?php _e('Das Archiv speichert aggregierte und anonymisierte Formulardaten. So bleiben Statistiken erhalten, auch wenn Elementor-Einträge gelöscht oder aufgeräumt werden.', 'elementor-forms-statistics'); ?></p>
        <table class="form-table" role="presentation">
            <tr>
                <th scope="row"><?php _e('Status', 'elementor-forms-statistics'); ?></th>
                <td>
                    <?php if ($initialized && $table_exists) : ?>
                        <span class="dashicons dashicons-yes" style="color: #46b450;"></span>
                        <?php _e('Aktiv', 'elementor-forms-statistics'); ?>
                    <?php else : ?>
                        <span class="dashicons dashicons-warning" style="color: #d63638;"></span>
                        <?php _e('Noch nicht initialisiert', 'elementor-forms-statistics'); ?>
                    <?php endif; ?>
                </td>
            </tr>
            <tr>
                <th scope="row"><?php _e('Letzter vollständiger Import', 'elementor-forms-statistics'); ?></th>
                <td><?php echo esc_html($last_run_display); ?></td>
            </tr>
            <tr>
                <th scope="row"><?php _e('Letzte Synchronisierung', 'elementor-forms-statistics'); ?></th>
                <td><?php echo esc_html($last_sync_display); ?></td>
            </tr>
            <tr>
                <th scope="row"><?php _e('Gespeicherte Monate', 'elementor-forms-statistics'); ?></th>
                <td><?php echo esc_html(number_format_i18n($row_count)); ?></td>
            </tr>
            <tr>
                <th scope="row"><?php _e('Anzahl unterschiedlicher Formulare', 'elementor-forms-statistics'); ?></th>
                <td><?php echo esc_html(number_format_i18n($distinct_forms)); ?></td>
            </tr>
            <tr>
                <th scope="row"><?php _e('Summe aller Anfragen', 'elementor-forms-statistics'); ?></th>
                <td><?php echo esc_html(number_format_i18n($total_requests)); ?></td>
            </tr>
        </table>
        <p><?php _e('Nach der Initialisierung synchronisiert das Plugin automatisch neue Einträge in kurzen Abständen. Bei Bedarf kann der Prozess hier jederzeit erneut gestartet werden.', 'elementor-forms-statistics'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <?php wp_nonce_field('mdp_archive_import', 'mdp_archive_nonce'); ?>
            <input type="hidden" name="action" value="mdp_run_archive_import">
            <?php if ($initialized) : ?>
                <label>
                    <input type="checkbox" name="mdp_archive_reset" value="1">
                    <?php _e('Archiv vollständig neu aufbauen (überschreibt bestehende Daten mit aktuellem Elementor-Stand)', 'elementor-forms-statistics'); ?>
                </label>
                <p class="description"><?php _e('Nur aktivieren, wenn das Archiv beschädigt ist. Achtung: Gelöschte Elementor-Einträge gehen dabei verloren.', 'elementor-forms-statistics'); ?></p>
            <?php endif; ?>
            <?php submit_button($button_label); ?>
        </form>
        </div>
    </div>
    <?php
}

function mdp_email_settings_page_callback() {
    if (!current_user_can('edit_posts')) {
        wp_die(__('Sie haben keine Berechtigung, diese Seite zu sehen.', 'elementor-forms-statistics'));
    }

    $message = '';
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mdp_email_settings_nonce']) && wp_verify_nonce($_POST['mdp_email_settings_nonce'], 'mdp_save_email_settings')) {
        $email_interval = isset($_POST['email_interval']) ? sanitize_text_field($_POST['email_interval']) : 'monthly';
        $allowed_intervals = array('disabled', 'daily', 'weekly', 'monthly');
        if (!in_array($email_interval, $allowed_intervals, true)) {
            $email_interval = 'monthly';
        }

        $email_recipients_raw = isset($_POST['email_recipients']) ? wp_unslash($_POST['email_recipients']) : '';
        $email_recipient_lines = preg_split('/[\r\n,]+/', $email_recipients_raw);
        $clean_recipients = [];
        foreach ($email_recipient_lines as $recipient_line) {
            $recipient_line = trim($recipient_line);
            if ($recipient_line === '') {
                continue;
            }
            $recipient_email = sanitize_email($recipient_line);
            if ($recipient_email) {
                $clean_recipients[] = strtolower($recipient_email);
            }
        }

        $email_message_input = isset($_POST['email_message']) ? wp_unslash($_POST['email_message']) : '';
        $email_message_input = is_string($email_message_input) ? trim($email_message_input) : '';
        $email_message = $email_message_input !== '' ? wp_kses_post($email_message_input) : mdp_get_default_email_message();
        $email_subject = isset($_POST['email_subject']) ? sanitize_text_field(wp_unslash($_POST['email_subject'])) : '';
        if ($email_subject === '') {
            $email_subject = '📈 ' . __('Anfragen Statistik – %s', 'elementor-forms-statistics');
        }

        $export_link_label = isset($_POST['export_link_label']) ? sanitize_text_field(wp_unslash($_POST['export_link_label'])) : '';
        $email_send_time = isset($_POST['email_send_time']) ? sanitize_text_field($_POST['email_send_time']) : '08:00';
        if (!preg_match('/^([01]\d|2[0-3]):([0-5]\d)$/', $email_send_time)) {
            $email_send_time = '08:00';
        }
        $email_day_of_month = isset($_POST['email_day_of_month']) ? (int) $_POST['email_day_of_month'] : 1;
        if ($email_day_of_month < 1) {
            $email_day_of_month = 1;
        } elseif ($email_day_of_month > 31) {
            $email_day_of_month = 31;
        }
        $email_weekday = isset($_POST['email_weekday']) ? strtolower(sanitize_text_field($_POST['email_weekday'])) : 'monday';
        $weekday_choices = array_keys(mdp_get_email_weekday_choices());
        if (!in_array($email_weekday, $weekday_choices, true)) {
            $email_weekday = 'monday';
        }

        $include_attachment = isset($_POST['include_attachment']) && $_POST['include_attachment'] === '1' ? '1' : '0';

        update_option('mdp_efs_email_interval', $email_interval);
        update_option('mdp_efs_email_recipients', implode("\n", $clean_recipients));
        update_option('mdp_efs_email_message', $email_message);
        update_option('mdp_efs_email_subject', $email_subject);
        update_option('mdp_efs_email_time', $email_send_time);
        update_option('mdp_efs_email_day_of_month', $email_day_of_month);
        update_option('mdp_efs_email_weekday', $email_weekday);
        update_option('mdp_efs_export_link_label', $export_link_label);
        update_option('mdp_efs_include_attachment', $include_attachment);
        mdp_reset_stats_email_schedule();
        $message = __('Einstellungen gespeichert.', 'elementor-forms-statistics');
    }

    $email_interval = mdp_get_schedule_interval();
    $email_recipients_text = mdp_get_email_recipients_text();
    $email_message = mdp_get_email_message();
    $email_subject_template = mdp_get_email_subject_template();
    $email_send_time = mdp_get_email_send_time();
    $email_day_of_month = mdp_get_email_day_of_month();
    $email_weekday = mdp_get_email_weekday();
    $manual_status = isset($_GET['mdp_send_status']) ? sanitize_text_field($_GET['mdp_send_status']) : '';
    $weekday_choices = mdp_get_email_weekday_choices();
    $link_label = mdp_get_export_link_label();
    $include_attachment = mdp_should_include_stats_attachment();
    $show_time_row = $email_interval !== 'disabled';
    $show_weekly_row = ($email_interval === 'weekly');
    $show_monthly_row = ($email_interval === 'monthly');
    ?>
    <div class="wrap">
        <?php mdp_render_admin_section_styles(); ?>
        <div class="mdp-settings-wrapper">
        <h1><?php _e('Einstellungen E-Mail Versand', 'elementor-forms-statistics'); ?></h1>
        <?php if ($message) : ?>
            <div class="notice notice-success"><p><?php echo esc_html($message); ?></p></div>
        <?php endif; ?>
        <?php if ($manual_status === 'success') : ?>
            <div class="notice notice-success"><p><?php _e('Die Statistik wurde erfolgreich versendet.', 'elementor-forms-statistics'); ?></p></div>
        <?php elseif ($manual_status === 'error') : ?>
            <div class="notice notice-error"><p><?php _e('Die Statistik konnte nicht gesendet werden. Bitte Empfänger prüfen.', 'elementor-forms-statistics'); ?></p></div>
        <?php endif; ?>
        <form method="post">
            <?php wp_nonce_field('mdp_save_email_settings', 'mdp_email_settings_nonce'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="email_interval"><?php _e('Automatischer Versand', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <select name="email_interval" id="email_interval">
                            <option value="disabled" <?php selected($email_interval, 'disabled'); ?>><?php _e('Deaktiviert', 'elementor-forms-statistics'); ?></option>
                            <option value="daily" <?php selected($email_interval, 'daily'); ?>><?php _e('Täglich', 'elementor-forms-statistics'); ?></option>
                            <option value="weekly" <?php selected($email_interval, 'weekly'); ?>><?php _e('Wöchentlich', 'elementor-forms-statistics'); ?></option>
                            <option value="monthly" <?php selected($email_interval, 'monthly'); ?>><?php _e('Monatlich', 'elementor-forms-statistics'); ?></option>
                        </select>
                        <p class="description"><?php _e('Legt fest, wie oft die Statistik per E-Mail verschickt wird.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr class="mdp-schedule-field mdp-field-time"<?php echo $show_time_row ? '' : ' style="display:none;"'; ?>>
                    <th scope="row">
                        <label for="email_send_time"><?php _e('Uhrzeit für den Versand', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <input type="time" name="email_send_time" id="email_send_time" value="<?php echo esc_attr($email_send_time); ?>">
                        <p class="description"><?php _e('Die Statistik wird zur angegebenen Uhrzeit versendet.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr class="mdp-schedule-field mdp-field-weekly"<?php echo $show_weekly_row ? '' : ' style="display:none;"'; ?>>
                    <th scope="row">
                        <label for="email_weekday"><?php _e('Wochentag für den Versand', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <select name="email_weekday" id="email_weekday">
                            <?php foreach ($weekday_choices as $weekday_slug => $weekday_label) : ?>
                                <option value="<?php echo esc_attr($weekday_slug); ?>" <?php selected($email_weekday, $weekday_slug); ?>><?php echo esc_html($weekday_label); ?></option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php _e('Nur sichtbar bei wöchentlichem Versand.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr class="mdp-schedule-field mdp-field-monthly"<?php echo $show_monthly_row ? '' : ' style="display:none;"'; ?>>
                    <th scope="row">
                        <label for="email_day_of_month"><?php _e('Tag im Monat', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <input type="number" min="1" max="31" name="email_day_of_month" id="email_day_of_month" value="<?php echo esc_attr($email_day_of_month); ?>">
                        <p class="description"><?php _e('Bei unterschiedlichen Monatslängen wird automatisch der letzte verfügbare Tag verwendet.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="email_recipients"><?php _e('Empfänger der Statistik', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <textarea name="email_recipients" id="email_recipients" rows="4" class="large-text code"><?php echo esc_textarea($email_recipients_text); ?></textarea>
                        <p class="description"><?php _e('Eine oder mehrere E-Mail-Adressen, getrennt durch Zeilenumbrüche oder Kommas.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="email_subject"><?php _e('Betreffzeile', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <input type="text" name="email_subject" id="email_subject" value="<?php echo esc_attr($email_subject_template); ?>" class="regular-text">
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="email_message"><?php _e('E-Mail Text', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <textarea name="email_message" id="email_message" rows="6" class="large-text code"><?php echo esc_textarea($email_message); ?></textarea>
                        <p class="description"><?php _e('Der komplette Text für den E-Mail-Body.', 'elementor-forms-statistics'); ?></p>
                        <p class="description"><?php printf(
                            __('%s wird durch den Link zur Statistik ersetzt. Fehlt der Platzhalter, hängt das System den Link am Ende an.', 'elementor-forms-statistics'),
                            '<code>' . esc_html(mdp_get_stats_link_placeholder()) . '</code>'
                        ); ?></p>
                        <p class="description"><?php _e('%s = Website URL', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="export_link_label"><?php _e('Text Link', 'elementor-forms-statistics'); ?></label>
                    </th>
                    <td>
                        <input type="text" name="export_link_label" id="export_link_label" value="<?php echo esc_attr($link_label); ?>" class="regular-text">
                        <p class="description"><?php _e('Dieser Text ersetzt „Passwortfreien Statistik-Zugang“ im Link, falls du ihn anders benennen möchtest.', 'elementor-forms-statistics'); ?></p>
                    </td>
                </tr>
            </table>
            <?php submit_button(__('Speichern', 'elementor-forms-statistics')); ?>
        </form>
        <script>
        (function($){
            function mdpToggleScheduleFields() {
                var interval = $('#email_interval').val();
                $('.mdp-field-time').toggle(interval !== 'disabled');
                $('.mdp-field-weekly').toggle(interval === 'weekly');
                $('.mdp-field-monthly').toggle(interval === 'monthly');
            }
            $(document).on('change', '#email_interval', mdpToggleScheduleFields);
            $(document).ready(mdpToggleScheduleFields);
        })(jQuery);
        </script>
        <hr>
        <h2><?php _e('Statistik sofort versenden', 'elementor-forms-statistics'); ?></h2>
        <p><?php _e('Verschickt die aktuelle Statistik einmalig an die oben definierten Empfänger.', 'elementor-forms-statistics'); ?></p>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="mdp-send-now-form">
            <?php wp_nonce_field('mdp_send_stats_now', 'mdp_send_now_nonce'); ?>
            <input type="hidden" name="action" value="mdp_send_stats_now">
            <?php submit_button(__('Statistik jetzt senden', 'elementor-forms-statistics'), 'secondary', 'mdp_send_now'); ?>
        </form>
        <hr>
        <h2><?php _e('Statistik als HTML exportieren', 'elementor-forms-statistics'); ?></h2>
        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="mdp-export-form">
            <?php wp_nonce_field('mdp_export_html', 'mdp_export_nonce'); ?>
            <input type="hidden" name="action" value="mdp_export_stats_html">
            <?php submit_button(__('Als HTML exportieren', 'elementor-forms-statistics'), 'secondary'); ?>
        </form>
        </div>
    </div>
    <?php
}

// Enable translation functions
add_action('plugins_loaded', function() {
    load_plugin_textdomain('elementor-forms-statistics', false, dirname(plugin_basename(__FILE__)) . '/languages/');
});

/****** Show Elmentor Submissions to editors **/
if (!class_exists('ElementorFormSubmissionsAccess'))
{

    class ElementorFormSubmissionsAccess
    {
        /**
         * Determine if the current user belongs to one of the allowed roles (and is not an administrator).
         * @return bool
         */
        static function hasAllowedRole()
        {
            if (current_user_can('manage_options')) {
                return false;
            }
            return mdp_user_has_submission_access_role();
        }

        /**
         * This is called around line 849 of wp-includes/rest-api/class-wp-rest-server.php by the ajax request which loads the data
         * into the form submissions view for Elementor (see the add_menu_page below). The ajax request checks the user has
         * the manage_options permission in modules/forms/submissions/data/controller.php within the handler's permission_callback.
         * This overrides that, and also for the call to modules/forms/submissions/data/forms-controller.php (which fills the
         * Forms dropdown on the submissions page). By changing the $route check below, you could open up more pages to editors.
         * @param array [endpoints=>hanlders]
         * @return array [endpoints=>hanlders]
         */
        static function filterRestEndpoints($endpoints)
        {
            if (self::hasAllowedRole()) {
                error_reporting(0); // there are a couple of PHP notices which prevent the Ajax JSON data from loading
                foreach($endpoints as $route=>$handlers) //for each endpoint
                    if (strpos($route, '/elementor/v1/form') === 0) //it is one of the elementor endpoints forms, form-submissions or form-submissions/export
                        foreach($handlers as $num=>$handler) //loop through the handlers
                            if (is_array ($handler) && isset ($handler['permission_callback'])) //if this handler has a permission_callback
                                $endpoints[$route][$num]['permission_callback'] = function($request){return true;}; //handler always returns true to grant permission
            }
            return $endpoints;
        }

        /**
         * Add the submissions page to the admin menu on the left for allowed roles only, as administrators
         * can already see it.
         */
        static function addOptionsPage()
        {
            if (!self::hasAllowedRole()) return;
            add_menu_page('Anfragen', 'Anfragen', 'edit_posts', 'e-form-submissions', function(){echo '<div id="e-form-submissions"></div>';}, 'dashicons-list-view', 3);


        }

        /**
         * Hook up the filter and action. I can't check if they are an editor here as the wp_user_can function
         * is not available yet.
         */
        static function hookIntoWordpress()
        {
            add_filter ('rest_endpoints', array('ElementorFormSubmissionsAccess', 'filterRestEndpoints'), 1, 3);
            add_action ('admin_menu', array('ElementorFormSubmissionsAccess', 'addOptionsPage'));
        }
    }

    ElementorFormSubmissionsAccess::hookIntoWordpress();
} //a wrapper to see if the class already exists or not



/* json */
// REST-API-Route registrieren
add_action('rest_api_init', function () {
    register_rest_route('mdp/v1', '/submissions', [
        'methods' => 'GET',
        'callback' => 'mdp_get_submission_data',
        'permission_callback' => function () {
            return current_user_can('edit_posts'); // Zugriffsberechtigungen für Autoren und höher
        }
    ]);
});

function mdp_get_submission_data($request) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'e_submissions';
    $selected_form_id = sanitize_text_field($request->get_param('form_id'));
    if (!$selected_form_id) {
        $selected_form_id = sanitize_text_field($request->get_param('element_id'));
    }
    $selected_referer_title = sanitize_text_field($request->get_param('referer_title')); // Legacy fallback

    // Basisabfrage
    $sql = "SELECT YEAR(created_at_gmt) AS jahr, MONTH(created_at_gmt) AS monat, COUNT(*) AS anzahl_anfragen 
            FROM " . $table_name . " 
            WHERE `status` NOT LIKE '%trash%'";

    // Formular-Filter anwenden
    if ($selected_form_id) {
        $sql .= $wpdb->prepare(" AND `element_id` = %s", $selected_form_id);
    } elseif ($selected_referer_title) { // Legacy fallback to avoid breaking existing clients
        $sql .= $wpdb->prepare(" AND `referer_title` = %s", $selected_referer_title);
    }

    $sql .= mdp_get_email_exclusion_clause($table_name);

    $sql .= " GROUP BY jahr, monat ORDER BY jahr, monat";

    $results = $wpdb->get_results($sql);

    // Datenstruktur für JSON-Ausgabe vorbereiten
    $data = [];
    foreach ($results as $result) {
        $data[] = [
            'jahr' => $result->jahr,
            'monat' => $result->monat,
            'anzahl_anfragen' => $result->anzahl_anfragen,
        ];
    }

    return rest_ensure_response($data);
}
